import { graphql } from "gatsby";
import * as React from "react";
import {
  Background,
  BorderRadius,
  CoreText,
  Layout,
  ResponsiveWrapper,
  StyledLayout,
  TextType,
  Title,
  TitleSize,
} from "twitch-core-ui";
import { ReleaseNotesTile } from "../components/release-notes-tile";
import { GithubVersionReleaseConnection } from "../graphql-types";

export interface PublicProps {
  data: {
    versions: GithubVersionReleaseConnection;
  };
}

export default class ReleaseNotesPage extends React.Component<PublicProps, {}> {
  public render() {
    let errorNotice: JSX.Element | undefined;

    if (!this.props.data.versions.edges) {
      return <CoreText>No release notes found!</CoreText>;
    }

    if (
      this.props.data.versions.edges.find(
        e => !!(e.node && e.node.id.match(/\(empty\)/)),
      )
    ) {
      errorNotice = (
        <StyledLayout
          padding={{ x: 3, y: 2 }}
          background={Background.Alt2}
          borderRadius={BorderRadius.Medium}
          border
        >
          <CoreText>
            Unable to fetch GitHub data. Ensure that you are on a network that
            can access the Twitch Enterprise GitHub and the Github environment
            variables are set.
          </CoreText>
        </StyledLayout>
      );
    }

    return (
      <ResponsiveWrapper>
        <Layout padding={5}>
          <Layout margin={{ bottom: 2 }}>
            <Title size={TitleSize.ExtraLarge} type={TextType.H1}>
              Release Notes
            </Title>
          </Layout>
          {errorNotice
            ? errorNotice
            : this.props.data.versions.edges.map(
                item =>
                  item.node && (
                    <Layout margin={{ y: 5 }}>
                      <ReleaseNotesTile node={item.node} />
                    </Layout>
                  ),
              )}
        </Layout>
      </ResponsiveWrapper>
    );
  }
}

export const query = graphql`
  query ReleaseNotesPageQuery {
    versions: allGithubVersionRelease {
      edges {
        node {
          id
          data {
            tagName
            htmlUrl
            publishedAt
            body
          }
        }
      }
    }
  }
`;
