import { isRelativeUrl, prefixHtmlAssets, withPathPrefix } from ".";

const TEST_BASE_PATH = "/prefix";

describe("gatsby-remark-prefix-assets utils", () => {
  it("prefixHtmlAssets prefixes a single image", () => {
    const testAssetPath = `/assets/test-image.jpg`;
    const testHtml = `<img src="/assets/test-image.jpg" />`;

    expect(prefixHtmlAssets(testHtml, TEST_BASE_PATH)).toEqual(
      `<img src="${TEST_BASE_PATH}${testAssetPath}" />`,
    );
  });

  it("prefixHtmlAssets prefixes multiple images", () => {
    const testHtml = `<div><div><img src="/assets/test-image-1.jpg" /></div><div><img src="/assets/test-image-2.jpg" /></div></div>`;

    expect(prefixHtmlAssets(testHtml, TEST_BASE_PATH)).toEqual(
      `<div><div><img src="${TEST_BASE_PATH}/assets/test-image-1.jpg" /></div><div><img src="${TEST_BASE_PATH}/assets/test-image-2.jpg" /></div></div>`,
    );
  });

  it("prefixHtmlAssets prefixes audio and video tags", () => {
    const testHtml = `<video type="video/mp4" src="/assets/test-video.mp4"></video><audio controls src="/assets/test-audio.mp3">Audio fallback text.</audio>`;

    expect(prefixHtmlAssets(testHtml, TEST_BASE_PATH)).toEqual(
      `<video type="video/mp4" src="${TEST_BASE_PATH}/assets/test-video.mp4"></video><audio controls src="${TEST_BASE_PATH}/assets/test-audio.mp3">Audio fallback text.</audio>`,
    );
  });

  it("prefixHtmlAssets does not prefix absolute URLs", () => {
    const testHttpSrc = `https://design.internal.justin.tv/assets/test-image.jpg`;

    expect(
      prefixHtmlAssets(`<img src="${testHttpSrc}" />`, TEST_BASE_PATH),
    ).toEqual(`<img src="${testHttpSrc}" />`);
  });

  it("prefixHtmlAssets does not prefix path-relative urls", () => {
    const testPathRelativeSrc = `static/assets/test-image.jpg`;

    expect(
      prefixHtmlAssets(`<img src="${testPathRelativeSrc}" />`, TEST_BASE_PATH),
    ).toEqual(`<img src="${testPathRelativeSrc}" />`);
  });

  it("withPathPrefix correctly adds the supplied base path", () => {
    const testAssetPath = "/assets/test-image.jpg";
    const expected = `${TEST_BASE_PATH}${testAssetPath}`;

    expect(withPathPrefix(testAssetPath, TEST_BASE_PATH)).toEqual(expected);
    expect(withPathPrefix(`/${testAssetPath}`, TEST_BASE_PATH)).toEqual(
      expected,
    );
  });

  it("isRelativeUrl returns the correct boolean value", () => {
    expect(isRelativeUrl("/assets/test-image.jpg")).toBe(true);
    expect(
      isRelativeUrl("https://design.internal.justin.tv/assets/test-image.jpg"),
    ).toBe(false);
    expect(isRelativeUrl("assets/test-image.jpg")).toBe(false);
  });

  it("prefixHtmlAssets returns the html string when no media tags are present", () => {
    const testHtmlString = `<div><p>This HTML does not contain any media.</p></div>`;
    expect(prefixHtmlAssets(testHtmlString)).toBe(testHtmlString);
  });
});
