/**
 * Finds and replaces image, video, and audio `src` attributes with a prefixed
 * URL (if relative).
 * @param htmlString The node's HTML as a string.
 * @param basePath Provided by Gatsby from the `pathPrefix` or `assetPrefix` settings.
 */
export const prefixHtmlAssets = (htmlString: string, basePath?: string) => {
  if (
    htmlString.match(
      /<img([\w\W]+?)\/>|<img([\w\W]+?)\/>|<video([\w\W]+?)\<\/video>|<audio([\w\W]+?)\<\/audio>/,
    )
  ) {
    return htmlString.replace(/(?<=src=").*?(?=")/g, src =>
      isRelativeUrl(src) ? withPathPrefix(src, basePath || "") : src,
    );
  } else {
    return htmlString;
  }
};

/**
 * Rewrite URLs with the path prefix.
 * @param url
 * @param pathPrefix
 */
export const withPathPrefix = (url: string, pathPrefix: string) =>
  (pathPrefix + url).replace(/\/\//, `/`);

/**
 * Test if is relative url
 * @param url
 */
export const isRelativeUrl = (url: string) =>
  url && url.startsWith(`/`) && !url.startsWith(`//`);
