import * as Octokat from "octokat";
import { getProcessedNode, NodeType } from "../utils/get-processed-node";
import {
  emptyGithubCommitItem,
  emptyGithubVersionRelease,
  GatsySourceGithubArguments,
  GithubRepo,
} from "./";

exports.sourceNodes = async (
  { actions, reporter }: any,
  { user, repository, token, rootURL, baseDir }: GatsySourceGithubArguments,
) => {
  const { createNode } = actions;
  let repo: GithubRepo | undefined = undefined;
  let fetchError: string | undefined = undefined;

  if (!user || !repository || !rootURL || !baseDir) {
    throw "gatsby-source-github-enterprise is missing configuration options.";
  }

  // Check for the access token – set in environment variables.
  if (token) {
    const octo = new Octokat({ token, rootURL });
    repo = octo.repos(user, repository);
  } else {
    // Throw an error (to stop build) if running `gatsby build` without specifically setting
    // GATSBY_ENV variable to `ci`.
    if (
      process.env.GATSBY_ENV !== "ci" &&
      process.env.NODE_ENV === "production"
    ) {
      throw "gatsby-source-github-enterprise is missing a Github personal access token.";
    }

    fetchError =
      "Please set a Github personal access token in your .env file as: GITHUB_ACCESS_TOKEN=xxxxx";
  }

  // Get Release Notes
  // Response matches API: https://developer.github.com/v3/repos/releases/
  if (repo) {
    try {
      const result = await repo.releases.read();
      result.items.forEach((item: any) => {
        createNode(
          getProcessedNode(NodeType.GithubRelease, item.tagName, {
            data: item,
          }),
        );
      });
    } catch (error) {
      fetchError = error.message || "Problem with querying github relases API";
    }
  }

  // If we are unable to connect to the Github repo or another error is
  // encountered while fetching data:
  if (!repo || fetchError) {
    // If during build – fail the build.
    if (process.env.NODE_ENV === "production") {
      throw fetchError;
    }

    reporter.info(`could not fetch GitHub data – ${fetchError}`);

    // Gatsby needs at least one data point in order to build the GraphQL
    // structure; and without it, the GQL queries will fail on the front-end.
    // To get around this issue when this plugin fails to create real data
    // (due to a missing token or connection issues), we supply a "empty" node.
    createNode(
      getProcessedNode(NodeType.GithubRelease, null, {
        data: emptyGithubVersionRelease,
      }),
    );
    createNode(
      getProcessedNode(NodeType.GithubCommits, null, {
        component: "",
        items: [emptyGithubCommitItem],
      }),
    );

    return;
  }

  reporter.success("fetch GitHub data");
  return;
};
