import * as fs from "fs";
import * as path from "path";
import * as coreui from "twitch-core-ui";
import * as ts from "typescript";

// Must be relative to the compiled dir for this script
const ROOT_DIR = path.resolve(__dirname, "../../..");
const TEMP_DIR = path.resolve(__dirname, "tmp");
const TEMP_FILE = `${TEMP_DIR}/code.tsx`;

// The line where the actual code we are testing gets placed on
const LINE_NUM_IN_TEMP_FILE = 5;

/**
 * This plugin for `remark-lint-code` will compile and test code blocks in markdown files.
 *
 * This is specifically for Core UI usage examples because it will wrap the code example in
 * a render method and will add an import statement for all Core UI components and exports.
 *
 * This task writes out a temporary .tsx file and runs it through the Typescript compiler API.
 */
module.exports = (_OPTIONS: any) => {
  return (node: any, file: any) => {
    const lineOffset = node.position.start.line - LINE_NUM_IN_TEMP_FILE + 2;

    // Print this as a progress indicator because this check takes a long time
    // tslint:disable-next-line:no-console
    console.log(`Checking ${file.path}:${node.position.start.line}`);

    createJsxFileSync(node.value);

    diagnostics().forEach(diagnostic => {
      if (diagnostic.file === undefined || diagnostic.start === undefined) {
        return;
      }

      const message = ts.flattenDiagnosticMessageText(
        diagnostic.messageText,
        "\n",
      );
      const errorPosition = diagnostic.file.getLineAndCharacterOfPosition(
        diagnostic.start,
      );

      const pos = {
        start: {
          line: lineOffset + errorPosition.line,
          column: errorPosition.character,
        },
        end: {
          line: lineOffset + errorPosition.line,
          column: errorPosition.character + (diagnostic.length || 0),
        },
      };

      // Don't report an error if it's outside of the actual code example (i.e. ignore import statement in temp file)
      if (
        pos.start.line > node.position.start.line &&
        pos.end.line < node.position.end.line
      ) {
        file.message(message, pos);
      }
    });
  };
};

function getTsConfig() {
  const config = JSON.parse(
    fs.readFileSync(`${ROOT_DIR}/tsconfig.json`, "utf8"),
  );
  const parsedConfig = ts.convertCompilerOptionsFromJson(
    config.compilerOptions,
    ROOT_DIR,
  );

  parsedConfig.options.outDir = `${TEMP_DIR}/out`;

  return parsedConfig;
}

/**
 * Compiles the temporary TS file and returns any diagnostics
 */
function diagnostics() {
  const config = getTsConfig();

  let program = ts.createProgram([TEMP_FILE], config.options);
  let emitResult = program.emit();

  let allDiagnostics = ts
    .getPreEmitDiagnostics(program)
    .concat(emitResult.diagnostics);

  return allDiagnostics;
}

/**
 * Writes a JSX file to disk which can be compiled by typescript;
 * This mimics what the react live preview is going to do when the site gets built.
 */
function createJsxFileSync(code: string) {
  if (!fs.existsSync(TEMP_DIR)) {
    fs.mkdirSync(TEMP_DIR);
  }

  const lines = [
    "import * as React from 'react';",
    getImportStatement(),
    "export default () => (",
    "<>",
    code,
    "</>",
    ");",
  ];
  fs.writeFileSync(TEMP_FILE, lines.join("\n"));
}

/**
 * Generates an import statement with all exported Core UI terms.
 *
 * Looks like:
 *     import { Accordion, AccordionBody, ..., ZIndex } from 'twitch-core-ui';
 */
function getImportStatement() {
  return Object.keys(coreui)
    .sort((a, b) => a.localeCompare(b, undefined, { sensitivity: "base" }))
    .reduce((value, current, i, all) => {
      if (i === 0) {
        value = `import { ${current}`;
      } else if (i < all.length - 1) {
        value = `${value}, ${current}`;
      } else {
        value = `${value}, ${current} } from 'twitch-core-ui'; \n`;
      }
      return value;
    }, "");
}
