import { MultiReporter, SpadeReporter } from 'tachyon-spade-reporter';
// tslint:disable-next-line
import type { Reporter, ReportSpadeEvent, SpadeEvent } from 'tachyon-spade-reporter';
// tslint:disable-next-line
import type { AllEvents } from './events'
/**
 * The named parameters passed in when constructing an <AppReporter>.
 * These fields are used to either augment any events reported or to
 * define the underlying reporter used to report events.
 */
export interface CommonProperties {
  platform: "web" | "android" | "ios" | string;
  app_version: string;
  language: string;
  organization_id: string;
  is_e2_whitelisted: boolean;
}

export interface ReporterOpts  {
  staticProps: CommonProperties;
  spadeUri: string;
  debug?: boolean;
}

export type FullEventType<T> = T & { properties: CommonProperties };

/*
* Decorator which attaches static properties before calling underlying spade reporters
*/
export class AppReporter
  implements Reporter<SpadeEvent> {
  /**
   * Base immutable properties for all events reported to spade.
   */
  public readonly staticProperties: CommonProperties;
  /**
   * Reporter to which this reporter delegates event reporting.
   */
  private reporter: Reporter;

  /**
   * Constructs a new AppReporter with base properties.
   */
  constructor({
    spadeUri,
    debug,
    staticProps
  }: ReporterOpts ) {
    this.staticProperties = staticProps;

    if (debug) {
      this.reporter = new MultiReporter([
        // tslint:disable-next-line: no-console
        {reportEvent: (event : AllEvents, opts: Object) => console.log("new spade event:", event, opts)},
        new SpadeReporter(spadeUri),
      ]);
      return;
    }
    this.reporter = new MultiReporter([
      new SpadeReporter(spadeUri),
    ]);
  }

  /**
   * reportEvent takes an event of EventType, augments it with static
   * and dynamic events, and then reports it to the appropriate event
   * receivers.
   */
  public reportEvent: ReportSpadeEvent<AllEvents> = (event, options) => {
    const fullEvent = {
      ...event,
      properties: {
        ...this.staticProperties,
        ...(event.properties || {}),
      },
    } as FullEventType<AllEvents>;
    this.reporter.reportEvent(fullEvent, options);
  };

}
