import { extractMarkdownLinksFromString } from "./parse-markdown-from-string";

describe("extractMarkdownLinksFromString", () => {
  it("extracts a single link", () => {
    const example =
      "Deprecated in favor of the new component [See New Component](https://www.example.com)";
    const result = extractMarkdownLinksFromString(example);

    expect(result.links.length).toBe(1);

    expect(result.links).toContainEqual(
      expect.objectContaining({
        props: {
          linkTo: "https://www.example.com",
          children: "See New Component",
        },
      }),
    );
  });

  it("extracts multiple links", () => {
    const example =
      "[Hello World](hello.com), welcome to [Twitch](www.twitch.tv)";
    const result = extractMarkdownLinksFromString(example);

    expect(result.links.length).toBe(2);

    expect(result.links).toContainEqual(
      expect.objectContaining({
        props: {
          linkTo: "hello.com",
          children: "Hello World",
        },
      }),
    );

    expect(result.links).toContainEqual(
      expect.objectContaining({
        props: {
          linkTo: "www.twitch.tv",
          children: "Twitch",
        },
      }),
    );
  });

  it("removes the links from the text", () => {
    const example =
      "Deprecated [in favor of](google.com) the new component [See New Component](https://www.example.com)";
    const result = extractMarkdownLinksFromString(example);
    expect(result.text).toBe("Deprecated  the new component");
  });

  it("has no limit by default", () => {
    const example =
      "[foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com)";
    const result = extractMarkdownLinksFromString(example);
    expect(result.links.length).toBe(7);
  });

  it("can limit the number of matches", () => {
    const example =
      "[foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com)";
    const result = extractMarkdownLinksFromString(example, 3);
    expect(result.links.length).toBe(3);
  });

  it("only removes the extracted links from the text", () => {
    const example =
      "[foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com)";
    const result = extractMarkdownLinksFromString(example, 3);
    expect(result.text).toBe(
      "[foo](bar.com) [foo](bar.com) [foo](bar.com) [foo](bar.com)",
    );
  });
});
