#!/bin/bash
set -e

show_usage_and_exit() {
  echo "Find ECS Task ARNs that are running on an Ec2 instance ID."
  echo "  Usage:   ./scripts/find_task_by_ec2.sh <dev|prod> <instance_id>"
  echo "  Example: ./scripts/find_task_by_ec2.sh dev i-018e96f45c451f50e"
  exit 1
}

export AWS_DEFAULT_REGION="us-west-2"

# param environment <dev|prod>
case "$1" in
	"dev")
    export AWS_PROFILE="twitch-eml-dev"
		CLUSTER="DevThresholdService-ClusterEB0386A7-1WWWPEGJO3JU5"
    ;;
  "prod")
    export AWS_PROFILE="twitch-eml-prod"
		CLUSTER="DevThresholdService-EC2ServiceF0CE72D0-169NHVHK8KUAFUpdate"
    ;;
  *)
    show_usage_and_exit
    ;;
esac

# param instance_id
[[ -z "$2" ]] && { echo "ERROR: instance_id not specified (e.g. i-018e96f45c451f50e)"; show_usage_and_exit; }
INSTANCE_ID="$2"

# Find ECS Instance ARN for that Ec2 instance ID.
echo ">> aws ecs list-container-instances where ec2InstanceId==$INSTANCE_ID"
AWS_OUT=$(aws ecs list-container-instances --cluster=$CLUSTER --filter="ec2InstanceId==$INSTANCE_ID")
ECS_INSTANCE_ARN=$(echo $AWS_OUT | jq '.containerInstanceArns[0]' --raw-output)
[[ "$ECS_INSTANCE_ARN" == "null" ]] && { echo "ECS Container Instance not found with Ec2 instance_id: $INSTANCE_ID"; exit 1; }

# Find task or tasks that running in that container instance
echo ">> aws ecs list-tasks --container-instance=$ECS_INSTANCE_ARN"
aws ecs list-tasks --cluster=$CLUSTER --container-instance=$ECS_INSTANCE_ARN | jq '.'
