// tslint:disable-next-line: no-submodule-imports
import 'source-map-support/register';

import * as ec2 from '@aws-cdk/aws-ec2';
import * as cdk from '@aws-cdk/core';

import { AlarmsStack } from './stacks/alarms';
import { CertificateStack } from './stacks/certificate-stack';
import { CloudmapStack } from './stacks/cloudmap-stack';
import {
  CommonProps,
  Ec2ScalingProps,
  GreeterScalingProps,
  SourceScalingProps,
  TaskScalingProps,
} from './stacks/common-props';
import { GreeterServiceStack } from './stacks/greeter-service-stack';
import { InterfaceVpcEndpointStack } from './stacks/interface-vpc-endpoint-stack';
import { LogGroupStack } from './stacks/loggroup-stack';
import { PathfinderServiceStack } from './stacks/pathfinder-service-stack';
import { RepositoryStack } from './stacks/repository-stack';
import { S3BucketStack } from './stacks/s3-bucket-stack';
import { SecretsStack } from './stacks/secrets-stack';
import { SourceServiceStack } from './stacks/source-service-stack';
import { ThresholdServiceStack } from './stacks/threshold-service-stack';
import { VpcStack } from './stacks/vpc-stack';
import { ZonesStack } from './stacks/zones-stack';

export interface ValidatorProps extends ec2.InterfaceVpcEndpointOptions {
  domainName: string;
}

export interface EnvironmentProps {
  common: CommonProps;
  greeter: GreeterScalingProps;
  pathfinder: TaskScalingProps;
  source: SourceScalingProps;
  threshold: Ec2ScalingProps;
  validator: ValidatorProps;
  alarms: {
    pagerdutyUrgentUrl: string;
    pagerdutyModerateUrl: string;
  } | null;
}

const app = new cdk.App();

// Develop Environment ("dev")
createStacksForEnvironment({
  common: {
    envName: 'Dev',
    env: {
      account: '565915620853',
      region: 'us-west-2',
    },
    internalDomainName: 'dev.eml.twitch.a2z.com',
    logLevel: 'info',
  },
  alarms: null, // no alarms needed in staging
  greeter: {
    cpu: 512,
    memoryLimitMiB: 1024, // 1GB
    taskCount: 1,
    authMethods: ['validator', 's2s'],
  },
  pathfinder: {
    cpu: 2048,
    memoryLimitMiB: 4096,
    taskCount: 3, // >= 3, must be an odd number
  },
  source: {
    instanceType: new ec2.InstanceType('c5.large'), // has 2 vCPU
    cpu: 1024, // 1 vCPU
    memoryLimitMiB: 1024, // 1GB
    taskCount: 1,
    asgCapacity: 2,
    rateLimitCount: '11',
    rateLimitPeriod: '5s',
  },
  threshold: {
    instanceType: new ec2.InstanceType('c5.large'), // has 2 vCPU
    cpu: 1024, // 1 vCPU
    memoryLimitMiB: 1024, // 1GB
    taskCount: 1,
    asgCapacity: 2,
  },
  validator: {
    domainName: 'extensions-validator-staging.internal.justin.tv',
    service: { name: 'com.amazonaws.vpce.us-west-2.vpce-svc-0bda7390fcc6cf65b', port: 443 },
    privateDnsEnabled: false,
  },
});

// Production Environment ("prod")
createStacksForEnvironment({
  common: {
    envName: 'Prod',
    env: {
      account: '342135511598',
      region: 'us-west-2',
    },
    internalDomainName: 'prod.eml.twitch.a2z.com',
    logLevel: 'info',
  },
  alarms: {
    pagerdutyUrgentUrl: 'https://events.pagerduty.com/integration/d8c6fa9c0c4549fca5a780db0410509b/enqueue',
    pagerdutyModerateUrl: 'https://events.pagerduty.com/integration/decc76477d034661a6082459e19ee91d/enqueue',
  },
  greeter: {
    cpu: 512,
    memoryLimitMiB: 1024, // 1GB
    taskCount: 3,
    authMethods: ['validator', 's2s'],
  },
  pathfinder: {
    cpu: 2048,
    memoryLimitMiB: 4096,
    taskCount: 3, // >= 3, must be an odd number
  },
  source: {
    instanceType: new ec2.InstanceType('c5.large'), // has 2 vCPU
    cpu: 1024, // 1 vCPU
    memoryLimitMiB: 1024, // 1GB
    taskCount: 6,
    asgCapacity: 9, // +50% to allow deploy in 2 batches
    rateLimitCount: '11',
    rateLimitPeriod: '5s',
  },
  threshold: {
    instanceType: new ec2.InstanceType('c5.large'), // has 2 vCPU
    cpu: 1024, // 1 vCPU
    memoryLimitMiB: 1024, // 1GB
    taskCount: 9,
    asgCapacity: 15, // +50% to allow deploy in 2 batches
  },
  validator: {
    domainName: 'extensions-validator.prod.us-west2.justin.tv',
    service: { name: 'com.amazonaws.vpce.us-west-2.vpce-svc-0797f09cd5e86bcdb', port: 443 },
    privateDnsEnabled: false,
  },
});

function createStacksForEnvironment(props: EnvironmentProps) {
  cdk.Tag.add(app, 'project', 'eml');

  const vpcStack = new VpcStack(app, props.common);
  const vpc = vpcStack.vpc;

  const secrets = new SecretsStack(app, props.common);

  const nspaceStack = new CloudmapStack(app, { ...props.common, domainName: 'eml', vpc });
  const nspace = nspaceStack.namespace;

  const zone = new ZonesStack(app, props.common).internalZone;

  const validator = new InterfaceVpcEndpointStack(app, {
    serviceName: 'Validator',
    vpc,
    ...props.common,
    ...props.validator,
  });

  // public load balancer
  const thresholdCert = new CertificateStack(app, { ...props.common, subdomain: 'connect', zone });

  // inter-service load balancers
  const greeterHostCert = new CertificateStack(app, { ...props.common, subdomain: 'greeter-host', zone });
  const pathfinderHostCert = new CertificateStack(app, { ...props.common, subdomain: 'pathfinder-host', zone });
  const pathfinderClientCert = new CertificateStack(app, {
    ...props.common,
    subdomain: 'pathfinder-client',
    zone,
  });

  // Greeter
  const GREETER = 'Greeter';
  const greeterRepo = new RepositoryStack(app, GREETER, props.common);
  const greeterLogs = new LogGroupStack(app, GREETER, props.common);
  const greeterService = new GreeterServiceStack(app, {
    ...props.common,
    ...props.greeter,
    serviceName: GREETER,
    repo: greeterRepo.repo,
    logs: greeterLogs.logGroup,
    authBinding: {
      target: validator.endpoint,
      url: new cdk.StringConcat().join('https://', validator.domainName),
      port: props.validator.service.port,
    },
    hostBinding: {
      certificateArn: greeterHostCert.cert.certificateArn,
      domainName: greeterHostCert.domainName,
      port: 443,
    },
    vpc,
    zone,
    secrets,
    nspace,
  });

  // Pathfinder
  const PATHFINDER = 'Pathfinder';
  const pathfinderRepo = new RepositoryStack(app, PATHFINDER, props.common);
  const pathfinderLogs = new LogGroupStack(app, PATHFINDER, props.common);
  const pathfinderService = new PathfinderServiceStack(app, {
    ...props.common,
    ...props.pathfinder,
    serviceName: PATHFINDER,
    repo: pathfinderRepo.repo,
    logs: pathfinderLogs.logGroup,
    clientBinding: {
      certificateArn: pathfinderClientCert.cert.certificateArn,
      domainName: pathfinderClientCert.domainName,
      port: 443,
    },
    hostBinding: {
      certificateArn: pathfinderHostCert.cert.certificateArn,
      domainName: pathfinderHostCert.domainName,
      port: 443,
    },
    vpc,
    zone,
    secrets,
    nspace,
  });

  // Threhsold
  const THRESHOLD = 'Threshold';
  const autoprofS3Bucket = new S3BucketStack(app, 'ThresholdAutoprofBucket', {
    ...props.common,
    bucketName: 'autoprof-e2ml-threshold-us-west-2', // https://wiki.twitch.com/display/VID/Autoprof
  });
  const thresholdRepo = new RepositoryStack(app, THRESHOLD, props.common);
  const thresholdLogs = new LogGroupStack(app, THRESHOLD, props.common);
  const thresholdService = new ThresholdServiceStack(app, {
    ...props.common,
    ...props.threshold,
    serviceName: THRESHOLD,
    repo: thresholdRepo.repo,
    logs: thresholdLogs.logGroup,
    hostUrl: new cdk.StringConcat().join('wss://', greeterHostCert.domainName),
    clientUrl: new cdk.StringConcat().join('https://', pathfinderClientCert.domainName),
    audienceBinding: {
      certificateArn: thresholdCert.cert.certificateArn,
      domainName: thresholdCert.domainName,
      port: 443,
    },
    autoprofBucket: autoprofS3Bucket.bucket,
    vpc,
    zone,
    secrets,
  });

  // Source
  const SOURCE = 'Source';
  const sourceRepo = new RepositoryStack(app, SOURCE, props.common);
  const sourceLogs = new LogGroupStack(app, SOURCE, props.common);
  const sourceService = new SourceServiceStack(app, {
    ...props.common,
    ...props.source,
    serviceName: SOURCE,
    repo: sourceRepo.repo,
    logs: sourceLogs.logGroup,
    hostUrl: new cdk.StringConcat().join('wss://', pathfinderHostCert.domainName),
    clientSecurityGroup: thresholdService.securityGroup,
    vpc,
    secrets,
  });

  // CloudWatch alarms and Pagerduty SNS topics
  if (props.alarms) {
    new AlarmsStack(app, 'Alarms', {
      ...props.common,
      pagerdutyUrgentUrl: props.alarms.pagerdutyUrgentUrl,
      pagerdutyModerateUrl: props.alarms.pagerdutyModerateUrl,
      serviceMetrics: [
        {
          name: GREETER,
          cpuUtilization: greeterService.service.metricCpuUtilization(),
          memUtilization: greeterService.service.metricMemoryUtilization(),
        },
        {
          name: PATHFINDER,
          cpuUtilization: pathfinderService.service.metricCpuUtilization(),
          memUtilization: pathfinderService.service.metricMemoryUtilization(),
        },
        {
          name: THRESHOLD,
          cpuUtilization: thresholdService.service.metricCpuUtilization(),
          memUtilization: thresholdService.service.metricMemoryUtilization(),
        },
        {
          name: SOURCE,
          cpuUtilization: sourceService.service.metricCpuUtilization(),
          memUtilization: sourceService.service.metricMemoryUtilization(),
        },
      ],
    });
  }
}
