import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as cwactions from '@aws-cdk/aws-cloudwatch-actions';
import * as sns from '@aws-cdk/aws-sns';
import * as subs from '@aws-cdk/aws-sns-subscriptions';
import * as cdk from '@aws-cdk/core';

import { CommonProps } from './common-props';

interface AlarmsStackProps extends CommonProps {
  pagerdutyUrgentUrl: string;
  pagerdutyModerateUrl: string;
  serviceMetrics: ServiceProps[];
}

interface ServiceProps {
  name: string;
  cpuUtilization: cloudwatch.Metric;
  memUtilization: cloudwatch.Metric;
}

export class AlarmsStack extends cdk.Stack {
  public pagerdutyUrgentTopic: sns.ITopic;
  public pagerdutyModerateTopic: sns.ITopic;

  constructor(scope: cdk.Construct, name: string, props: AlarmsStackProps) {
    super(scope, `${props.envName}${name}`, props);
    const envName = props.envName.toLowerCase();

    // SNS Topics that send an https request to the CloudWatch integration
    this.pagerdutyUrgentTopic = new sns.Topic(this, 'PagerdutyUrgentTopic');
    this.pagerdutyUrgentTopic.addSubscription(new subs.UrlSubscription(props.pagerdutyUrgentUrl));
    this.pagerdutyModerateTopic = new sns.Topic(this, 'PagerdutyModerateTopic');
    this.pagerdutyModerateTopic.addSubscription(new subs.UrlSubscription(props.pagerdutyModerateUrl));

    // CloudWatch Alarms
    // -----------------

    for (const svc of props.serviceMetrics) {
      // CPU
      this.reportUrgent(
        new cloudwatch.Alarm(this, `Alarm${svc.name}CPU`, {
          alarmDescription: `E2ML${svc.name} ECS service CPU utilization >= 80%`,
          metric: svc.cpuUtilization,
          period: cdk.Duration.minutes(5),
          evaluationPeriods: 3,
          threshold: 80,
        })
      );

      // Memory
      this.reportUrgent(
        new cloudwatch.Alarm(this, `Alarm${svc.name}Memory`, {
          alarmDescription: `E2ML${svc.name} ECS service Memory utilization >= 80%`,
          metric: svc.memUtilization,
          period: cdk.Duration.minutes(5),
          evaluationPeriods: 3,
          threshold: 80,
        })
      );
    }

    // Greeter Authorizations
    this.reportModerate(
      new cloudwatch.Alarm(this, 'AlarmGreeterInactiveAuths', {
        alarmDescription: 'E2ML Greeter is not reporting authorizations.',
        metric: svcMetric(envName, 'E2MLGreeter', 'count.broker.auth;action:validate,result:ok'),
        period: cdk.Duration.minutes(5),
        evaluationPeriods: 3,
        threshold: 0,
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
      })
    );

    // Pathfinder Elections
    this.reportModerate(
      new cloudwatch.Alarm(this, 'AlarmPathfinderElectionFailures', {
        alarmDescription:
          'E2ML Pathfinder is failing 50% of elections (voting to assign a new address to a Source). Election failures should not happen, please do a full restart',
        metric: new cloudwatch.MathExpression({
          expression: '100 * fail / (fail + complete)', // failure pertentage
          usingMetrics: {
            fail: svcMetric(envName, 'E2MLPathfinder', 'count.broker.elections;action:fail'),
            complete: svcMetric(envName, 'E2MLPathfinder', 'count.broker.elections;action:complete'),
          },
        }),
        evaluationPeriods: 3,
        threshold: 50,
      })
    );

    // Threshold Fanout Count
    this.reportModerate(
      new cloudwatch.Alarm(this, 'AlarmThresholdFanoutCount', {
        alarmDescription: 'E2ML Threshold is not responding messages back to clients.',
        metric: svcMetric(envName, 'E2MLThreshold', 'count.threshold.fanout.count'),
        period: cdk.Duration.minutes(5),
        evaluationPeriods: 3,
        threshold: 0,
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
      })
    );

    // Source Bytes written (responses)
    this.reportModerate(
      new cloudwatch.Alarm(this, 'AlarmSourceBytesWritten', {
        alarmDescription: 'E2ML Source is not writing any messages back to listeners.',
        metric: svcMetric(envName, 'E2MLSource', 'count.audience.bytes.written'),
        period: cdk.Duration.minutes(5),
        evaluationPeriods: 3,
        threshold: 0,
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
      })
    );

    // Source Status Reports
    this.reportModerate(
      new cloudwatch.Alarm(this, 'AlarmSourceStatusReports', {
        alarmDescription: 'E2ML Source is not reporting status updates to Pathfinder.',
        metric: svcMetric(envName, 'E2MLSource', 'count.host.reports'),
        period: cdk.Duration.minutes(5),
        evaluationPeriods: 3,
        threshold: 0,
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
      })
    );
  }

  private reportModerate(alarm: cloudwatch.Alarm) {
    alarm.addAlarmAction(new cwactions.SnsAction(this.pagerdutyModerateTopic));
  }

  private reportUrgent(alarm: cloudwatch.Alarm) {
    alarm.addAlarmAction(new cwactions.SnsAction(this.pagerdutyUrgentTopic));
  }
}

function svcMetric(stage: string, svcName: string, metric: string): cloudwatch.Metric {
  return new cloudwatch.Metric({
    namespace: svcName,
    metricName: metric,
    dimensions: {
      Region: 'us-west-2',
      Service: svcName,
      Stage: stage,
    },
  });
}
