import * as certificatemanager from '@aws-cdk/aws-certificatemanager';
import * as route53 from '@aws-cdk/aws-route53';
import * as cdk from '@aws-cdk/core';

import { CommonProps } from './common-props';
import * as utils from './utils';

interface CertificateStackProps extends CommonProps {
  subdomain: string;
  zone: route53.IHostedZone;
  external?: boolean;
}

export class CertificateStack extends cdk.Stack {
  public static createInactive(scope: cdk.Construct, props: CertificateStackProps): cdk.Stack {
    const description = 'Placeholder for a certificate stack, used to force CDK to cleanup resources';
    return new cdk.Stack(scope, CertificateStack.createId(props), { description, ...props });
  }

  public static createId(props: CertificateStackProps): string {
    return `${props.envName}Certificate${utils.titleCase(props.subdomain)}`;
  }

  public static isActive(stack: cdk.Stack): stack is CertificateStack {
    return (stack as CertificateStack).cert !== undefined;
  }

  public readonly cert: certificatemanager.ICertificate;
  public readonly domainName: string;

  constructor(scope: cdk.Construct, props: CertificateStackProps) {
    super(scope, CertificateStack.createId(props), props);
    cdk.Tag.add(this, 'environment', props.envName.toLowerCase());
    const parent = props.external ? props.externalDomainName || props.internalDomainName : props.internalDomainName;
    this.domainName = `${props.subdomain}.${parent}`;
    this.cert = new certificatemanager.DnsValidatedCertificate(this, 'Certficate', {
      domainName: this.domainName,
      hostedZone: props.zone,
    });
  }
}
