import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as lbv2 from '@aws-cdk/aws-elasticloadbalancingv2';
import * as iam from '@aws-cdk/aws-iam';
import * as awslogs from '@aws-cdk/aws-logs';
import * as route53 from '@aws-cdk/aws-route53';
import * as route53targets from '@aws-cdk/aws-route53-targets';
import * as servicediscovery from '@aws-cdk/aws-servicediscovery';
import * as cdk from '@aws-cdk/core';

import { CommonProps, GreeterScalingProps } from './common-props';
import { SecretsStack } from './secrets-stack';

const DISCOVERY_PORT = 8000; // UNUSED, this was for validation auth to connect form the public
const STATUS_PORT = 3000; // UNUSED, this was for validation auth flow for Threshold
const PEERING_PORT = 3010; // UNUSED, this was for validation auth flow, for multiple greeters acting like pathfinder peers

interface GreeterServiceStackProps extends CommonProps, GreeterScalingProps {
  secrets: SecretsStack;
  serviceName: string;
  authBinding: {
    target: ec2.IConnectable;
    url: string;
    port: number;
  };
  hostBinding: {
    certificateArn: string;
    domainName: string;
    port: number;
  };
  nspace: servicediscovery.IPrivateDnsNamespace;
  repo: ecr.IRepository;
  logs: awslogs.ILogGroup;
  vpc: ec2.Vpc;
  zone: route53.IHostedZone;
}

export class GreeterServiceStack extends cdk.Stack {
  public readonly service: ecs.FargateService;
  public readonly securityGroup: ec2.SecurityGroup;

  constructor(scope: cdk.Construct, props: GreeterServiceStackProps) {
    const account = props.env!.account!;
    const envName = props.envName.toLowerCase();
    const serviceName = props.serviceName.toLowerCase();

    super(scope, `${props.envName}${props.serviceName}Service`, props);
    cdk.Tag.add(this, 'environment', envName);
    cdk.Tag.add(this, 'service', serviceName);

    // Cluster
    const cluster = new ecs.Cluster(this, 'Cluster', {
      vpc: props.vpc,
      containerInsights: true,
    });

    // Task Definition
    const taskDefinition = new ecs.FargateTaskDefinition(this, 'FTask', {
      cpu: props.cpu,
      family: `${serviceName}-fargate`,
      memoryLimitMiB: props.memoryLimitMiB,
    });
    taskDefinition.addToTaskRolePolicy(
      // Send custom metrics to CloudWatch for TwitchTelemetry
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        resources: ['*'],
      })
    );

    // Service container
    const container = taskDefinition.addContainer(serviceName, {
      image: ecs.ContainerImage.fromEcrRepository(props.repo, 'latest'),
      command: ['/bin/sh', '/root/fargate.sh'], // set container name, host ip dynamically
      stopTimeout: cdk.Duration.seconds(10),
      dockerLabels: {
        account,
        'com.docker.compose.service': serviceName,
      },
      environment: {
        CLIENT_AUTH_METHODS: props.authMethods.join(','),
        PEER_DISCOVERY_DATA: '', // unused, this was for reservation flow
        LOG: props.logLevel,
        METRICS_ACCOUNT: account,
        METRICS_METHOD: 'twitchtelemetry',
        METRICS_SERVICE: serviceName,
        VALIDATOR_URL: props.authBinding.url,
        ENV_NAME: envName,
      },
      secrets: {
        CLIENT_S2S_SECRET: ecs.Secret.fromSecretsManager(props.secrets.greeterClient),
        HOST_S2S_SECRET: ecs.Secret.fromSecretsManager(props.secrets.greeterHost),
        PEER_S2S_SECRET: ecs.Secret.fromSecretsManager(props.secrets.greeterPeer),
      },
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'eml',
        logGroup: props.logs,
        datetimeFormat: '%Y/%m/%d %H:%M:%S',
      }),
    });
    container.addPortMappings(
      { containerPort: PEERING_PORT }, // placed first for cloud map use
      { containerPort: STATUS_PORT },
      { containerPort: DISCOVERY_PORT }
    );
    container.addUlimits({
      name: ecs.UlimitName.NOFILE,
      softLimit: 10240,
      hardLimit: 10240,
    });

    // Security group to allow greeter to call ECS Validator
    this.securityGroup = new ec2.SecurityGroup(this, 'ServiceSG', {
      vpc: props.vpc,
      allowAllOutbound: true,
    });
    this.securityGroup.connections.allowTo(props.authBinding.target, ec2.Port.tcp(props.authBinding.port));
    // for NLB healh check; restrict access to private subnets or NLB ENI primary ips once available in CDK
    this.securityGroup.addIngressRule(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(STATUS_PORT));

    // ECS Service
    this.service = new ecs.FargateService(this, 'Fargate', {
      cluster,
      desiredCount: props.taskCount,
      taskDefinition,
      minHealthyPercent: 60,
      maxHealthyPercent: 200,
      securityGroup: this.securityGroup,
    });

    // Load Balancer
    const loadBalancer = new lbv2.NetworkLoadBalancer(this, 'HostNlb', {
      vpc: props.vpc,
      internetFacing: false,
    });

    // Load Balancer Listener
    const listener = loadBalancer.addListener('Listener', {
      port: props.hostBinding.port,
      certificates: [{ certificateArn: props.hostBinding.certificateArn }],
      protocol: lbv2.Protocol.TLS,
    });

    // Load Balancer Target Group
    const targetGroup = listener.addTargets('Target', {
      port: STATUS_PORT,
      deregistrationDelay: cdk.Duration.seconds(0), // unregister immediately from balancing, does not affect existing connections
    });
    targetGroup.addTarget(
      this.service.loadBalancerTarget({
        containerName: serviceName,
        containerPort: STATUS_PORT,
      })
    );

    // DNS Record
    new route53.ARecord(this, 'HostDns', {
      zone: props.zone,
      recordName: props.hostBinding.domainName,
      target: route53.AddressRecordTarget.fromAlias(new route53targets.LoadBalancerTarget(loadBalancer)),
    });
  }
}
