import * as ec2 from '@aws-cdk/aws-ec2';
import * as route53 from '@aws-cdk/aws-route53';
import * as cdk from '@aws-cdk/core';

import { CommonProps } from './common-props';
import { InterfaceVpcEndpointTarget } from './interface-vpc-endpoint-target';
import * as utils from './utils';

export interface InterfaceVpcEndpointStackProps extends CommonProps, ec2.InterfaceVpcEndpointOptions {
  serviceName: string;
  domainName: string;
  vpc: ec2.IVpc;
}

export class InterfaceVpcEndpointStack extends cdk.Stack {
  public readonly endpoint: ec2.InterfaceVpcEndpoint;
  public readonly domainName: string;
  public readonly zone: route53.IHostedZone;

  constructor(scope: cdk.Construct, props: InterfaceVpcEndpointStackProps) {
    super(scope, `${props.envName}${utils.titleCase(props.serviceName)}InterfaceVpcEndpoint`, props);

    cdk.Tag.add(this, 'environment', props.envName.toLowerCase());
    cdk.Tag.add(this, 'service', props.serviceName.toLowerCase());

    this.domainName = props.domainName; // determined by origin of the service endpoint
    this.endpoint = new ec2.InterfaceVpcEndpoint(this, 'Endpoint', props);
    this.zone = new route53.PrivateHostedZone(this, 'Zone', {
      zoneName: props.domainName,
      vpc: props.vpc,
    });

    new route53.ARecord(this, 'CName', {
      zone: this.zone,
      target: route53.AddressRecordTarget.fromAlias(new InterfaceVpcEndpointTarget(this.endpoint)),
    });
  }
}
