import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as lbv2 from '@aws-cdk/aws-elasticloadbalancingv2';
import * as iam from '@aws-cdk/aws-iam';
import * as awslogs from '@aws-cdk/aws-logs';
import * as route53 from '@aws-cdk/aws-route53';
import * as route53targets from '@aws-cdk/aws-route53-targets';
import * as servicediscovery from '@aws-cdk/aws-servicediscovery';
import * as cdk from '@aws-cdk/core';

import { CommonProps, TaskScalingProps } from './common-props';
import { SecretsStack } from './secrets-stack';

const DISCOVERY_PORT = 8000; // for Threshold
const STATUS_PORT = 3001; // for Sources
const PEERING_PORT = 3011; // for other Pathfinders

interface PathfinderServiceBinding {
  certificateArn: string;
  domainName: string;
  port: number;
}

interface PathfinderServiceStackProps extends CommonProps, TaskScalingProps {
  secrets: SecretsStack;
  serviceName: string;
  clientBinding: PathfinderServiceBinding;
  hostBinding: PathfinderServiceBinding;
  repo: ecr.IRepository;
  nspace: servicediscovery.IPrivateDnsNamespace;
  logs: awslogs.ILogGroup;
  vpc: ec2.Vpc;
  zone: route53.IHostedZone;
}

export class PathfinderServiceStack extends cdk.Stack {
  public readonly service: ecs.FargateService;
  public readonly securityGroup: ec2.SecurityGroup;

  constructor(scope: cdk.Construct, props: PathfinderServiceStackProps) {
    const serviceName = props.serviceName.toLowerCase();
    const envName = props.envName.toLowerCase();
    const account = props.env!.account!;

    super(scope, `${props.envName}${props.serviceName}Service`, props);
    cdk.Tag.add(this, 'environment', envName);
    cdk.Tag.add(this, 'service', serviceName);

    // Cluster
    const cluster = new ecs.Cluster(this, 'Cluster', {
      vpc: props.vpc,
      containerInsights: true,
    });

    // Task Definition
    const taskDefinition = new ecs.FargateTaskDefinition(this, 'FTask', {
      cpu: props.cpu,
      family: `${serviceName}-fargate`,
      memoryLimitMiB: props.memoryLimitMiB,
    });
    taskDefinition.addToTaskRolePolicy(
      // Send custom metrics to CloudWatch for TwitchTelemetry
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        resources: ['*'],
      })
    );

    // Service container
    const container = taskDefinition.addContainer(serviceName, {
      image: ecs.ContainerImage.fromEcrRepository(props.repo, 'latest'),
      command: ['/bin/sh', '/root/fargate.sh'], // set container name, host ip dynamically
      stopTimeout: cdk.Duration.seconds(10),
      dockerLabels: {
        account,
        'com.docker.compose.service': serviceName,
      },
      environment: {
        LOG: props.logLevel,
        METRICS_ACCOUNT: account,
        METRICS_METHOD: 'twitchtelemetry',
        METRICS_SERVICE: serviceName,
        ENV_NAME: envName,
        PEER_DISCOVERY_DATA: 'pathfinder.eml', // cloud map private DNS entry
        PEER_DISCOVERY_METHOD: 'dns',
      },
      secrets: {
        CLIENT_S2S_SECRET: ecs.Secret.fromSecretsManager(props.secrets.pathfinderClient),
        HOST_S2S_SECRET: ecs.Secret.fromSecretsManager(props.secrets.pathfinderHost),
        PEER_S2S_SECRET: ecs.Secret.fromSecretsManager(props.secrets.pathfinderPeer),
      },
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'eml',
        logGroup: props.logs,
        datetimeFormat: '%Y/%m/%d %H:%M:%S',
      }),
      healthCheck: {
        command: ['CMD-SHELL', `nc -z localhost ${DISCOVERY_PORT}`],
        interval: cdk.Duration.seconds(10),
      },
    });
    container.addPortMappings(
      { containerPort: PEERING_PORT }, // placed first for cloud map use
      { containerPort: STATUS_PORT },
      { containerPort: DISCOVERY_PORT }
    );
    container.addUlimits({
      name: ecs.UlimitName.NOFILE,
      softLimit: 10240,
      hardLimit: 10240,
    });

    // Pathfinder has special deployment rules:
    //  1) always leave at least one instance running (special case when min capacity == 1)
    //  2) drop instance total instead of increasing it to preserve quorum mechanics/elections in progress
    if (props.taskCount < 3 || props.taskCount % 2 === 0) {
      throw new Error(
        `Invalid pathfinder props.taskCount = ${props.taskCount}; must be >= 3 and odd, so election votes can resolve`
      );
    }
    const taskCount = props.taskCount;
    const minHealthyPercent = taskCount > 1 ? Math.floor((100 * (taskCount - 1)) / taskCount) : 100;
    const maxHealthyPercent = taskCount > 1 ? 100 : 200;

    // Service's security group, allow other Pathfinder peers to connect directly
    this.securityGroup = new ec2.SecurityGroup(this, 'ServiceSG', {
      vpc: props.vpc,
      allowAllOutbound: true,
    });
    this.securityGroup.connections.allowInternally(ec2.Port.tcp(PEERING_PORT));
    // for host NLB healh check; restrict access to private subnets (or NLB ENI primary ips once available in CDK)
    this.securityGroup.addIngressRule(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(STATUS_PORT));

    // ECS Service
    this.service = new ecs.FargateService(this, 'Fargate', {
      cluster,
      cloudMapOptions: {
        name: 'pathfinder',
        dnsTtl: cdk.Duration.seconds(5),
        cloudMapNamespace: props.nspace,
      },
      desiredCount: taskCount,
      taskDefinition,
      minHealthyPercent,
      maxHealthyPercent,
      securityGroup: this.securityGroup,
    });

    // Client Load Balancer security Group
    const clientLBSecurityGroup = new ec2.SecurityGroup(this, 'ClientAlbSg', {
      vpc: props.vpc,
    });
    clientLBSecurityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(props.clientBinding.port));

    // Client Load Balancer, this is where Threshold connects to request available sources
    const clientLoadBalancer = new lbv2.ApplicationLoadBalancer(this, 'ClientAlb', {
      vpc: props.vpc,
      internetFacing: false,
      securityGroup: clientLBSecurityGroup,
    });
    const clientListener = clientLoadBalancer.addListener('Listener', {
      port: props.clientBinding.port,
      certificateArns: [props.clientBinding.certificateArn],
      open: false,
    });
    const clientTargetGroup = clientListener.addTargets('Target', {
      port: DISCOVERY_PORT,
      deregistrationDelay: cdk.Duration.seconds(10),
      healthCheck: {
        path: '/health',
        healthyHttpCodes: '204',
        timeout: cdk.Duration.seconds(4),
      },
    });
    clientTargetGroup.addTarget(
      this.service.loadBalancerTarget({
        containerName: serviceName,
        containerPort: DISCOVERY_PORT,
      })
    );
    new route53.ARecord(this, 'ClientDns', {
      zone: props.zone,
      recordName: props.clientBinding.domainName,
      target: route53.AddressRecordTarget.fromAlias(new route53targets.LoadBalancerTarget(clientLoadBalancer)),
    });

    const binding = props.hostBinding;
    const containerName = props.serviceName.toLowerCase();

    // Host Load Balancer, this is where Sources connect to declare themselves available
    const hostLoadBalancer = new lbv2.NetworkLoadBalancer(this, 'HostNlb', {
      vpc: props.vpc,
      internetFacing: false,
    });
    const hostListener = hostLoadBalancer.addListener('Listener', {
      port: binding.port,
      certificates: [{ certificateArn: binding.certificateArn }],
      protocol: lbv2.Protocol.TLS,
    });
    const hostTargetGroup = hostListener.addTargets('Target', {
      port: STATUS_PORT,
      deregistrationDelay: cdk.Duration.seconds(0), // unregister immediately from balancing, does not affect existing connections
    });
    hostTargetGroup.addTarget(this.service.loadBalancerTarget({ containerName, containerPort: STATUS_PORT }));

    // DNS Record
    new route53.ARecord(this, 'HostDns', {
      zone: props.zone,
      recordName: binding.domainName,
      target: route53.AddressRecordTarget.fromAlias(new route53targets.LoadBalancerTarget(hostLoadBalancer)),
    });
  }
}
