import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as iam from '@aws-cdk/aws-iam';
import * as awslogs from '@aws-cdk/aws-logs';
import * as cdk from '@aws-cdk/core';

import { CommonProps, SourceScalingProps } from './common-props';
import { SecretsStack } from './secrets-stack';

const SERVICE_PORT = 3003;

interface SourceServiceStackProps extends CommonProps, SourceScalingProps {
  secrets: SecretsStack;
  serviceName: string;
  hostUrl: string;
  clientSecurityGroup: ec2.SecurityGroup;
  logs: awslogs.ILogGroup;
  vpc: ec2.Vpc;
  repo: ecr.IRepository;
}

export class SourceServiceStack extends cdk.Stack {
  public readonly service: ecs.Ec2Service;
  public readonly securityGroup: ec2.SecurityGroup;

  constructor(scope: cdk.Construct, props: SourceServiceStackProps) {
    const account = props.env!.account!;
    const envName = props.envName.toLowerCase();
    const serviceName = props.serviceName.toLowerCase();

    super(scope, `${props.envName}${props.serviceName}Service`, props);
    cdk.Tag.add(this, 'environment', envName);
    cdk.Tag.add(this, 'service', serviceName);

    // Cluster
    const cluster = new ecs.Cluster(this, 'Cluster', {
      vpc: props.vpc,
      containerInsights: true,
    });
    cluster.addCapacity('ASG', {
      instanceType: props.instanceType,
      minCapacity: props.asgCapacity,
      maxCapacity: props.asgCapacity,
    });

    // Task Definition
    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'ETask', {
      family: `${serviceName}-ec2`,
      networkMode: ecs.NetworkMode.AWS_VPC,
    });
    taskDefinition.addToTaskRolePolicy(
      // Send custom metrics to CloudWatch for TwitchTelemetry
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        resources: ['*'],
      })
    );

    // Service container
    const container = taskDefinition.addContainer(serviceName, {
      image: ecs.ContainerImage.fromEcrRepository(props.repo, 'latest'),
      command: ['/bin/sh', '/root/ecs-ec2.sh'], // set container name, host ip dynamically
      cpu: props.cpu,
      memoryLimitMiB: props.memoryLimitMiB,
      stopTimeout: cdk.Duration.seconds(10),
      dockerLabels: {
        account,
        'com.docker.compose.service': serviceName,
      },
      environment: {
        HOST_URL: props.hostUrl,
        LOG: props.logLevel,
        METRICS_ACCOUNT: account,
        METRICS_SERVICE: serviceName,
        METRICS_METHOD: 'twitchtelemetry',
        ENV_NAME: envName,
        RATE_LIMIT_COUNT: props.rateLimitCount,
        RATE_LIMIT_PERIOD: props.rateLimitPeriod,
      },
      secrets: {
        HOST_S2S_SECRET: ecs.Secret.fromSecretsManager(props.secrets.pathfinderHost),
      },
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'eml',
        logGroup: props.logs,
        datetimeFormat: '%Y/%m/%d %H:%M:%S',
      }),
      healthCheck: {
        command: ['CMD-SHELL', `nc -z localhost ${SERVICE_PORT}`],
        interval: cdk.Duration.seconds(10),
      },
    });
    container.addPortMappings({ containerPort: SERVICE_PORT });
    container.addUlimits({
      name: ecs.UlimitName.NOFILE,
      softLimit: 102400,
      hardLimit: 102400,
    });

    // Security group to allow connections from Threshold
    this.securityGroup = new ec2.SecurityGroup(this, 'ServiceSG', {
      vpc: props.vpc,
      allowAllOutbound: true,
    });
    this.securityGroup.connections.allowFrom(props.clientSecurityGroup, ec2.Port.tcp(SERVICE_PORT));

    // ECS Service
    this.service = new ecs.Ec2Service(this, `EC2`, {
      cluster,
      desiredCount: props.taskCount,
      minHealthyPercent: props.minHealthyPercent ?? 100,
      maxHealthyPercent: props.maxHealthyPercent ?? 200,
      taskDefinition,
      securityGroup: this.securityGroup,
      placementConstraints: [ecs.PlacementConstraint.distinctInstances()],
    });
  }
}
