import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as lbv2 from '@aws-cdk/aws-elasticloadbalancingv2';
import * as iam from '@aws-cdk/aws-iam';
import * as awslogs from '@aws-cdk/aws-logs';
import * as route53 from '@aws-cdk/aws-route53';
import * as route53targets from '@aws-cdk/aws-route53-targets';
import * as s3 from '@aws-cdk/aws-s3';
import * as cdk from '@aws-cdk/core';

import { CommonProps, Ec2ScalingProps } from './common-props';
import { SecretsStack } from './secrets-stack';

const SERVICE_PORT = 3002;

// TEMPORARY
interface ThresholdServiceBinding {
  certificateArn: string;
  domainName: string;
  port: number;
}

interface ThresholdServiceStackProps extends CommonProps, Ec2ScalingProps {
  secrets: SecretsStack;
  serviceName: string;
  hostUrl: string;
  clientUrl: string;
  audienceBinding: ThresholdServiceBinding;
  logs: awslogs.ILogGroup;
  vpc: ec2.Vpc;
  zone: route53.IHostedZone;
  repo: ecr.IRepository;
  autoprofBucket: s3.IBucket;
}

export class ThresholdServiceStack extends cdk.Stack {
  public readonly service: ecs.Ec2Service;
  public readonly securityGroup: ec2.SecurityGroup;

  constructor(scope: cdk.Construct, props: ThresholdServiceStackProps) {
    const account = props.env!.account!;
    const envName = props.envName.toLowerCase();
    const serviceName = props.serviceName.toLowerCase();

    super(scope, `${props.envName}${props.serviceName}Service`, props);
    cdk.Tag.add(this, 'environment', envName);
    cdk.Tag.add(this, 'service', serviceName);

    // Cluster
    const cluster = new ecs.Cluster(this, 'Cluster', {
      vpc: props.vpc,
      containerInsights: true,
    });
    cluster.addCapacity('ASG', {
      instanceType: props.instanceType,
      minCapacity: props.asgCapacity,
      maxCapacity: props.asgCapacity,
    });

    // Task Definition
    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'ETask', {
      family: `${serviceName}-ec2`,
      networkMode: ecs.NetworkMode.AWS_VPC,
    });
    taskDefinition.addToTaskRolePolicy(
      // Send custom metrics to CloudWatch for TwitchTelemetry
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        resources: ['*'],
      })
    );

    // Service container
    const container = taskDefinition.addContainer(serviceName, {
      image: ecs.ContainerImage.fromEcrRepository(props.repo, 'latest'),
      command: ['/bin/sh', '/root/ecs-ec2.sh'], // set container name, host ip dynamically
      cpu: props.cpu,
      memoryLimitMiB: props.memoryLimitMiB,
      stopTimeout: cdk.Duration.seconds(10),
      dockerLabels: {
        account,
        'com.docker.compose.service': serviceName,
      },
      environment: {
        CLIENT_URL: props.clientUrl,
        HOST_URL: props.hostUrl,
        HOST_NAME: `wss://${props.audienceBinding.domainName}`,
        LOG: props.logLevel,
        AUTOPROF_BUCKET: props.autoprofBucket.bucketName,
        METRICS_ACCOUNT: account,
        METRICS_METHOD: 'twitchtelemetry',
        METRICS_SERVICE: serviceName,
        ENV_NAME: envName,
      },
      secrets: {
        CLIENT_S2S_SECRET: ecs.Secret.fromSecretsManager(props.secrets.pathfinderClient),
        HOST_S2S_SECRET: ecs.Secret.fromSecretsManager(props.secrets.greeterHost),
      },
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'eml',
        logGroup: props.logs,
        datetimeFormat: '%Y/%m/%d %H:%M:%S',
      }),
      healthCheck: {
        command: ['CMD-SHELL', `nc -z localhost ${SERVICE_PORT}`],
        interval: cdk.Duration.seconds(10),
      },
    });
    container.addPortMappings({ containerPort: SERVICE_PORT });
    container.addUlimits({
      name: ecs.UlimitName.NOFILE,
      softLimit: 102400,
      hardLimit: 102400,
    });

    // Security group to allow public access (NOTE: I think this was done for the unused validation auth flow and should be removed for security)
    this.securityGroup = new ec2.SecurityGroup(this, 'ServiceSG', {
      vpc: props.vpc,
      allowAllOutbound: true,
    });
    this.securityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(SERVICE_PORT));

    // ECS Service
    this.service = new ecs.Ec2Service(this, `EC2`, {
      cluster,
      desiredCount: props.taskCount,
      minHealthyPercent: props.minHealthyPercent ?? 100,
      maxHealthyPercent: props.maxHealthyPercent ?? 200,
      taskDefinition,
      securityGroup: this.securityGroup,
      placementConstraints: [ecs.PlacementConstraint.distinctInstances()],
    });

    // Load Balancer
    const loadBalancer = new lbv2.NetworkLoadBalancer(this, 'ClientNlb', {
      vpc: props.vpc,
      internetFacing: true,
    });
    const listener = loadBalancer.addListener('Listener', {
      port: props.audienceBinding.port,
      certificates: [{ certificateArn: props.audienceBinding.certificateArn }],
      protocol: lbv2.Protocol.TLS,
    });
    const targetGroup = listener.addTargets('Target', {
      port: SERVICE_PORT,
      deregistrationDelay: cdk.Duration.seconds(1), // unregister immediately from balancing, does not affect existing connections
    });
    targetGroup.addTarget(
      this.service.loadBalancerTarget({
        containerName: serviceName,
        containerPort: SERVICE_PORT,
      })
    );
    new route53.ARecord(this, 'ClientDns', {
      zone: props.zone,
      recordName: props.audienceBinding.domainName,
      target: route53.AddressRecordTarget.fromAlias(new route53targets.LoadBalancerTarget(loadBalancer)),
    });

    // Allow task to upload Autoprof profiles to the S3 bucket
    props.autoprofBucket.grantWrite(taskDefinition.taskRole); // s3:PutObject to upload autprof profiles

    // Allow Autoprof service to read from the S3 bucket
    const autoprofRole = new iam.Role(this, 'AutoprofReaderRole', {
      assumedBy: new iam.ArnPrincipal('arn:aws:iam::636562298954:role/autoprof-reader'),
    });
    autoprofRole.addToPolicy(
      new iam.PolicyStatement({
        actions: ['s3:GetObject'],
        resources: [props.autoprofBucket.arnForObjects('*')],
      })
    );
    autoprofRole.addToPolicy(
      new iam.PolicyStatement({
        actions: ['s3:ListBucket', 's3:GetBucketLocation', 's3:GetBucketNotification', 's3:PutBucketNotification'],
        resources: [props.autoprofBucket.bucketArn],
      })
    );
  }
}
