import { Address } from '../address';
import { AddressBuilder } from '../address_builder';
import * as errors from '../errors';
import { CompoundAddress } from './compound_address';
import { SimpleAddress } from './simple_address';
import { SingleFilterAddress } from './single_filter_address';

const nspaceReg = /[-._a-zA-Z0-9]+/;

export class AddressBuilderImpl implements AddressBuilder {
  private version: number = 0;
  private namespace: string = '';
  private filters: Map<string, string> = new Map<string, string>();

  public withNamespace(namespace: string): AddressBuilder {
    this.namespace = namespace;
    return this;
  }

  public withVersion(version: number): AddressBuilder {
    this.version = version;
    return this;
  }

  public withFilter(param: string, value: string): AddressBuilder {
    this.filters.set(param, value);
    return this;
  }

  public build(): Address {
    if (!this.version) {
      throw errors.MissingVersion;
    }
    if (this.version < 1 || this.version > 255) {
      throw errors.IllegalVersion;
    }
    if (this.namespace.length < 1) {
      throw errors.MissingNamespace;
    }
    if (!nspaceReg.test(this.namespace)) {
      throw errors.IllegalNamespace;
    }
    switch (this.filters.size) {
      case 0:
        return new SimpleAddress(this.namespace, this.version);
      case 1:
        const entry = this.filters.entries().next().value;
        return new SingleFilterAddress(this.namespace, this.version, entry[0], entry[1]);
      default:
        return new CompoundAddress(this.namespace, this.version, this.filters);
    }
  }
}
