import { Address } from '../address';
import { AddressScope } from '../address_scope';
import * as errors from '../errors';
import { SingleFilterAddress } from './single_filter_address';

function parents2(namespace: string, version: number, filters: Map<string, string>): AddressScope[] {
  const parents: AddressScope[] = [];
  for (const [param, value] of filters) {
    parents.push(new SingleFilterAddress(namespace, version, param, value));
  }
  return parents;
}

function parentsN(namespace: string, version: number, filters: Map<string, string>): AddressScope[] {
  const parents: AddressScope[] = [];
  for (const [param] of filters) {
    const copy = new Map(filters);
    copy.delete(param);
    parents.push(new CompoundAddress(namespace, version, copy));
  }
  return parents;
}

export class CompoundAddress implements Address {
  public readonly key: string;
  public readonly parents: AddressScope[];
  constructor(
    readonly namespace: string,
    readonly version: number,
    private readonly filters: Map<string, string>,
  ) {
    if (filters.size < 2) {
      throw errors.UseBuilder;
    }

    // build canonical key from filters
    const fragments: string[] = [];
    for (const [param, value] of this.filters) {
      fragments.push(`${param}=${value}`);
    }
    fragments.sort();
    this.key = `${this.namespace}@${this.version}?${fragments.join('&')}`;
    this.parents = filters.size > 2 ?
      parentsN(namespace, version, filters) :
      parents2(namespace, version, filters);
  }

  public get cardinality() { return this.filters.size + 1; }
  public filter(param: string) { return this.filters.get(param); }
  public includes(addr: Address) {
    for (const [param, value] of this.filters) {
      if (addr.filter(param) !== value) {
        return false;
      }
    }
    return addr.namespace === this.namespace && addr.version === this.version;
  }
}
