import * as WebSocketClient from 'universal-websocket-client';
import { Address } from '../address';
import { LogFunction, LogLevel } from '../log_function';
import { Position } from '../position';
import { Segment } from '../segment';
import { Source } from '../source';
import { DiscoveryResult } from './discovery_result';
import * as header from './header';
import * as marshal from './marshal';
import * as opcode from './opcode';
import { RequestIdFactory } from './request_id_factory';
import { ResponseHandler } from './response_handler';
import { Tracker } from './tracker';
import * as unmarshal from './unmarshal';
import * as utf8 from './utf8';

type HostClosedFunction = (binding: Binding) => void;
type HostDrainedFunction = (binding: Binding) => void;
type HostExpiringFunction = (binding: Binding) => void;
type ClosedFunction = (addr: Address, src: Source, endPosition: number) => void;
type LostFunction = (addr: Address, src: Source, at: Segment) => void;
type SentFunction = (addr: Address, src: Source, at: Segment, content: ArrayBuffer) => void;
type MessageFunction = (reqId: number) => ArrayBuffer | null;

// TODO : timeout stale handlers
export class Binding {
  private readonly reqs: RequestIdFactory = new RequestIdFactory();
  private readonly handlers: Map<number, ResponseHandler<boolean | Position>> = new Map<number, ResponseHandler<boolean | Position>>();
  private readonly messages: ArrayBuffer[] = [];
  private readonly ws: WebSocketClient;

  constructor(
    readonly hostUrl: string,
    private readonly onSent: SentFunction,
    private readonly onLost: LostFunction,
    private readonly onClosed: ClosedFunction,
    private readonly onHostExpiring: HostExpiringFunction,
    private readonly onHostClosed: HostClosedFunction,
    private readonly onHostDrained: HostDrainedFunction,
    private readonly log: LogFunction,
  ) {
    log(LogLevel.Info, 'stream binding', this.hostUrl);
    this.ws = new WebSocketClient(hostUrl);
    this.ws.binaryType = 'arraybuffer';
    this.ws.onerror = this.onError.bind(this);
    this.ws.onopen = this.onOpen.bind(this);
    this.ws.onclose = this.onClose.bind(this);
    this.ws.onmessage = this.onRawMessage.bind(this);
  }

  public initialize(result: DiscoveryResult): Promise<boolean> {
    return this.queue<boolean>('Init', (reqId) => {
      const buffer = new ArrayBuffer(7 + utf8.byteLength(result.accessCode));
      const bytes = new Uint8Array(buffer);
      bytes[6] = result.method;
      if (header.set(bytes, opcode.Init, reqId) && utf8.marshal(result.accessCode, bytes, 7)) {
        return buffer;
      }
      return null;
    });
  }

  public refresh(result: DiscoveryResult): Promise<boolean> {
    return this.queue<boolean>('Refresh', (reqId) => {
      const buffer = new ArrayBuffer(7 + utf8.byteLength(result.accessCode));
      const bytes = new Uint8Array(buffer);
      bytes[6] = result.method;
      if (header.set(bytes, opcode.Refresh, reqId) && utf8.marshal(result.accessCode, bytes, 7)) {
        return buffer;
      }
      reqId = reqId;
      return null;
    });
  }

  public join(addr: Address, tracker: Tracker | undefined): Promise<boolean> {
    return this.queue<boolean>('Join', (reqId) => {
      const buffer = new ArrayBuffer(6 + 8 + utf8.byteLength(addr.key));
      const bytes = new Uint8Array(buffer);
      const src = tracker ? tracker.src : undefined;
      const pos = tracker ? tracker.at : 0;
      if (header.set(bytes, opcode.Join, reqId) &&
        marshal.source(src, bytes, 6) &&
        marshal.position(pos, bytes, 10) &&
        marshal.address(addr, bytes, 14)) {
        return buffer;
      }
      return null;
    });
  }

  public part(addr: Address): Promise<boolean> {
    return this.queue<boolean>('Part', (reqId) => {
      const buffer = new ArrayBuffer(6 + utf8.byteLength(addr.key));
      const bytes = new Uint8Array(buffer);
      if (header.set(bytes, opcode.Part, reqId) && marshal.address(addr, bytes, 6)) {
        return buffer;
      }
      return null;
    });
  }

  public send(addr: Address, content: ArrayBuffer, isDelta: boolean): Promise<Position> {
    return this.queue<Position>('Send', (reqId) => {
      const addrLength = utf8.byteLength(addr.key);
      const buffer = new ArrayBuffer(8 + addrLength + content.byteLength);
      const bytes = new Uint8Array(buffer);
      if (!header.set(bytes, opcode.Send, reqId) || !marshal.address(addr, bytes, 7)) {
        return null;
      }
      bytes[6] = isDelta ? 1 : 0;
      bytes[7 + addrLength] = 0;
      let offset = 8 + addrLength;
      const src = new Uint8Array(content);
      for (let i = 0; i < src.length; ++i, ++offset) {
        bytes[offset] = src[i];
      }
      return buffer;
    });
  }

  public release(addr: Address): Promise<boolean> {
    return this.queue<boolean>('Release', (reqId) => {
      const buffer = new ArrayBuffer(6 + utf8.byteLength(addr.key));
      const bytes = new Uint8Array(buffer);
      if (header.set(bytes, opcode.Release, reqId) && marshal.address(addr, bytes, 6)) {
        return buffer;
      }
      return null;
    });
  }

  public close(): void {
    if (this.ws.readyState !== this.ws.CLOSED) {
      this.ws.close();
    } else {
      this.onHostClosed(this);
    }
  }

  private onOpen(): void {
    let msg = this.messages.shift();
    while (msg) {
      this.sendInternal(msg);
      msg = this.messages.shift();
    }
  }

  private onClose(ev: any): void {
    this.log(LogLevel.Info, 'Close', ev);
    this.onHostClosed(this);
  }

  private onRawMessage(ev: any): void {
    this.log(LogLevel.Debug, 'recv', ev.data);
    const bytes = new Uint8Array(ev.data);
    switch (header.getCode(bytes)) {
    case opcode.Drain: {
      this.onHostDrained(this);
      break;
    }
    case opcode.Ack: {
      const reqId = header.getRequest(bytes);
      const handler = this.getHandler(reqId);
      const src = unmarshal.source(bytes, 6);
      const pos = unmarshal.position(bytes, 10);
      if (handler) {
        handler.resolve(src && pos ? new Tracker(src, pos) : true);
      }
      break;
    }
    case opcode.Closed: {
      const src = unmarshal.source(bytes, 4);
      const pos = unmarshal.position(bytes, 8);
      const addr = unmarshal.address(bytes, 12);
      if (addr && src && pos) {
        this.onClosed(addr, src, pos);
      }
      break;
    }
    case opcode.Sent: {
      const src = unmarshal.source(bytes, 4);
      const segment = unmarshal.segment(bytes, 8);
      const addr = unmarshal.address(bytes, 16);
      if (addr && src && segment) {
        const n = utf8.end(bytes, 20);
        const end = ev.data.byteLength;
        const start = n ? n + 1 : end;
        this.onSent(addr, src, segment, ev.data.slice(start, end));
      }
      break;
    }
    case opcode.Error: {
      const errMsg = utf8.unmarshal(bytes, 6);
      this.log(LogLevel.Warning, errMsg);
      const reqId = header.getRequest(bytes);
      if (RequestIdFactory.isValid(reqId)) {
        const handler = this.getHandler(reqId);
        if (handler) {
          handler.reject(new Error(errMsg));
        }
      }
      break;
    }
    case opcode.Lost: {
      const src = unmarshal.source(bytes, 4);
      const segment = unmarshal.segment(bytes, 8);
      const addr = unmarshal.address(bytes, 20);
      if (addr && src && segment) {
        this.onLost(addr, src, segment);
      }
      break;
    }
    case opcode.Expiring: {
      this.onHostExpiring(this);
      break;
    }
    default:
      this.log(LogLevel.Warning, 'Unknown code:', header.getCode(bytes));
      break;
    }
  }

  private onError(ev: any) {
    this.log(LogLevel.Error, 'SocketError', ev);
  }

  private queue<T extends boolean | Position>(type: string, generator: MessageFunction): Promise<T> {
    const reqId = this.reqs.next;
    const msg = generator(reqId);
    if (!msg) {
      return Promise.reject(new Error(`Failed to serialize ${type}`));
    }
    const handler = new ResponseHandler<T>();
    this.handlers.set(reqId, handler);
    if (this.ws.readyState === this.ws.OPEN) {
      this.sendInternal(msg);
    } else {
      this.messages.push(msg);
    }
    return handler.promise;
  }

  private sendInternal(buffer: ArrayBuffer): void {
    this.log(LogLevel.Debug, 'send', buffer);
    this.ws.send(buffer);
  }

  private getHandler(reqId: number): ResponseHandler<Position | boolean> | undefined {
    const handler = this.handlers.get(reqId);
    if (handler) {
      this.handlers.delete(reqId);
    }
    return handler;
  }
}
