import { Address } from '../address';
import { Listener } from '../listener';
import { Position } from '../position';
import { Reader } from '../reader';
import { Segment } from '../segment';
import { Source } from '../source';
import { Writer } from '../writer';
import { Binding } from './binding';
import { DeferredClose } from './deferred_close';
import { DeferredMessage } from './deferred_message';
import { DeferredWrite } from './deferred_write';
import { ResponseHandler } from './response_handler';
import { Tracker } from './tracker';

export type hostRequest = (addr: Address, connect: boolean, binding: Binding | null) => void;

export class Channel implements Reader, Writer {
  private readonly listeners: Set<Listener> = new Set<Listener>();
  private readonly messages: DeferredWrite[] = [];
  private hasJoined: ResponseHandler<boolean> = new ResponseHandler<boolean>();
  private tracker?: Tracker; // used to track rejoin position
  private binding: Binding | null = null;
  private writers: number = 0;
  constructor(
    readonly address: Address,
    private readonly updateBinding: hostRequest,
    private readonly parents: Channel[],
  ) {}

  public send(content: ArrayBuffer, isDelta: boolean): Promise<Position> {
    if (this.binding) {
      return this.binding.send(this.address, content, isDelta);
    }
    const msg = new DeferredMessage(content, isDelta);
    this.messages.push(msg);
    this.requestBinding();
    return msg.result.promise;
  }

  public onWriterRequest(): Writer {
    this.writers++;
    return this;
  }

  public close(): Promise<boolean> {
    if (--this.writers !== 0) {
      return Promise.resolve(true);
    }

    if (this.binding) {
      const result = this.binding.release(this.address);
      if (!this.listeners.size) {
        this.updateBinding(this.address, false, this.binding);
      }
      return result;
    }
    // optimization: close without send while disconnected does not
    // require telling the service that the writer has closed; do
    // not connect to disconnect
    if (this.messages.length === 0) {
      return Promise.resolve(true);
    }

    const msg = new DeferredClose();
    this.messages.push(msg);
    this.requestBinding();

    // check if we can shut down the binding after the future close
    const channel = this;
    msg.result.promise.then(() => {
      if (channel.binding && !channel.writers && !channel.listeners.size) {
        channel.dropBinding();
      }
    });

    return msg.result.promise;
  }

  public join(listener: Listener): Promise<boolean> {
    if (this.listeners.has(listener)) {
      return this.hasJoined.promise.then(() => false);
    }
    this.listeners.add(listener);
    return this.tryJoin();
  }

  public leave(listener: Listener): Promise<boolean> {
    if (!this.listeners.has(listener)) {
      return Promise.resolve(false);
    }
    this.listeners.delete(listener);
    if (this.binding && !this.listeners.size) {
      const out = this.binding.part(this.address);
      if (!this.writers) {
        this.dropBinding();
      }
      return out;
    }
    return Promise.resolve(true);
  }

  public bind(binding: Binding | null): void {
    if (this.hasJoined.resolved) {
      this.hasJoined = new ResponseHandler<boolean>();
    }
    this.binding = binding;
    if (this.listeners.size) {
      this.tryJoin();
    }
    if (binding) {
      for (const msg of this.messages) {
        msg.execute(this.address, binding);
      }
      this.messages.length = 0;
    }
  }

  public onSent(from: Source, at: Segment, content: ArrayBuffer) {
    this.updateTracker(from, at);
    this.notify(this.address, from, at, content);
  }

  public onLost(from: Source, at: Segment) {
    this.updateTracker(from, at);
    this.notify(this.address, from, at, null);
  }

  public onClosed(from: Source, pos: number) {
    this.updateTracker(from, new Segment(0, pos));
    // todo : fix loss notification on close
    this.notifyClosed(this.address);
  }

  private updateTracker(from: Source, at: Segment) {
    if (this.tracker && this.tracker.src.id === from.id && this.tracker.at >= at.end) {
      return;
    }
    this.tracker = new Tracker(from, at.end);
  }

  private notifyClosed(address: Address) {
    for (const listener of this.listeners) {
      listener.onStreamClosed(address);
    }
    for (const parent of this.parents) {
      parent.notifyClosed(address);
    }
  }

  private notify(address: Address, from: Source, at: Segment, content: ArrayBuffer | null) {
    for (const listener of this.listeners) {
      const pos = listener.currentPosition(address);
      const start = pos && pos.src.id === from.id ? pos.at : 0;
      if (start < at.start) {
        listener.onDataLost(address, from, new Segment(start, at.start));
      }
      if (start < at.end) {
        if (content) {
          listener.onDataReceived(address, from, at, content);
        } else {
          listener.onDataLost(address, from, at);
        }
      }
    }
    for (const parent of this.parents) {
      parent.notify(address, from, at, content);
    }
  }

  private tryJoin(): Promise<boolean> {
    if (!this.binding) {
      this.requestBinding();
    } else {
      if (this.hasJoined.bound) {
        this.hasJoined = new ResponseHandler<boolean>();
      }
      this.hasJoined.bind(this.binding.join(this.address, this.tracker));
    }
    return this.hasJoined.promise;
  }

  private requestBinding(): void { this.updateBinding(this.address, true, this.binding); }
  private dropBinding(): void { this.updateBinding(this.address, false, this.binding); }
}
