import axios from 'axios';
import { Address } from '../address';
import { AuthSource } from '../auth_data';
import { LogFunction, LogLevel } from '../log_function';
import { DiscoveryResult } from './discovery_result';

export class Discovery {
  constructor(
    readonly hostUrl: string,
    private readonly auth: AuthSource,
    private readonly log: LogFunction,
  ) {
    if (!hostUrl.endsWith('/')) {
      hostUrl = hostUrl + '/';
    }
  }

  public update(hostUrl: string): DiscoveryResult {
    return DiscoveryResult.fromAuthData(hostUrl, this.auth());
  }

  public find(addr: Address): Promise<DiscoveryResult> {
    const authData = this.auth();
    this.log(LogLevel.Info, 'host request', this.hostUrl, addr.key);
    if (this.hostUrl.startsWith('ws')) {
      return Promise.resolve(DiscoveryResult.fromAuthData(this.hostUrl, authData));
    }
    return axios.get(`${this.hostUrl}v1/host`, {
      headers: {
        Authorization: `Bearer ${authData.token}`,
        'Client-Id': authData.clientId,
      },
      params: {
        address: addr.key,
      },
    }).then((response) => {
      return DiscoveryResult.fromHttpResult(response.data);
    });
  }
}
