import { Address } from '../address';
import * as marshal from './marshal';

describe('marshal.address', () => {
  it('marshals a standard address properly', () => {
    const addr = Address.parse('test@1?filter=value');
    const buffer = new ArrayBuffer(addr.key.length);
    const bytes = new Uint8Array(buffer);
    expect(marshal.address(addr, bytes, 0)).toBe(true);
    expect(bytes).toEqual(new Uint8Array(Buffer.from(addr.key)));
  });

  it('refuses to overrun the buffer', () => {
    const addr = Address.parse('test@1?filter=value');
    const buffer = new ArrayBuffer(addr.key.length - 2);
    const bytes = new Uint8Array(buffer);
    bytes[0] = 0xcd;
    expect(marshal.address(addr, bytes, 0)).toBe(false);
    expect(bytes[0]).toBe(0xcd);
  });
});

describe('marshal.position', () => {
  it('marshals min value (0x00000000) properly', () => {
    const buffer = new ArrayBuffer(4);
    const bytes = new Uint8Array(buffer);
    expect(marshal.position(0x00000000, bytes, 0)).toBe(true);
    expect(bytes[0]).toBe(0);
    expect(bytes[1]).toBe(0);
    expect(bytes[2]).toBe(0);
    expect(bytes[3]).toBe(0);
  });

  it('marshals max value (0xffffffff) properly', () => {
    const buffer = new ArrayBuffer(4);
    const bytes = new Uint8Array(buffer);
    expect(marshal.position(0xffffffff, bytes, 0)).toBe(true);
    expect(bytes[0]).toBe(0xff);
    expect(bytes[1]).toBe(0xff);
    expect(bytes[2]).toBe(0xff);
    expect(bytes[3]).toBe(0xff);
  });

  it('marshals as bigendian', () => {
    const buffer = new ArrayBuffer(4);
    const bytes = new Uint8Array(buffer);
    expect(marshal.position(0xaabbccdd, bytes, 0)).toBe(true);
    expect(bytes[0]).toBe(0xaa);
    expect(bytes[1]).toBe(0xbb);
    expect(bytes[2]).toBe(0xcc);
    expect(bytes[3]).toBe(0xdd);
  });

  it('marshals to an offset', () => {
    const buffer = new ArrayBuffer(5);
    const bytes = new Uint8Array(buffer);
    bytes[0] = 0xcd;
    expect(marshal.position(0xaabbccdd, bytes, 1)).toBe(true);
    expect(bytes[0]).toBe(0xcd);
    expect(bytes[1]).toBe(0xaa);
    expect(bytes[2]).toBe(0xbb);
    expect(bytes[3]).toBe(0xcc);
    expect(bytes[4]).toBe(0xdd);
  });

  it('refuses to overrun buffer', () => {
    const buffer = new ArrayBuffer(4);
    const bytes = new Uint8Array(buffer);
    bytes[0] = 0xcd;
    bytes[1] = 0xcd;
    bytes[2] = 0xcd;
    bytes[3] = 0xcd;
    expect(marshal.position(0x00000000, bytes, 1)).toBe(false);
    expect(bytes[0]).toBe(0xcd);
    expect(bytes[1]).toBe(0xcd);
    expect(bytes[2]).toBe(0xcd);
    expect(bytes[3]).toBe(0xcd);
  });
});

describe('marshal.source', () => {
  it('marshals undefined properly', () => {
    const buffer = new ArrayBuffer(4);
    const bytes = new Uint8Array(buffer);
    expect(marshal.source(undefined, bytes, 0)).toBe(true);
    expect(bytes[0]).toBe(0);
    expect(bytes[1]).toBe(0);
    expect(bytes[2]).toBe(0);
    expect(bytes[3]).toBe(0);
  });

  it('marshals min value (0x00000000) properly', () => {
    const buffer = new ArrayBuffer(4);
    const bytes = new Uint8Array(buffer);
    expect(marshal.source({ id: 0x00000000 }, bytes, 0)).toBe(true);
    expect(bytes[0]).toBe(0);
    expect(bytes[1]).toBe(0);
    expect(bytes[2]).toBe(0);
    expect(bytes[3]).toBe(0);
  });

  it('marshals max value (0xffffffff) properly', () => {
    const buffer = new ArrayBuffer(4);
    const bytes = new Uint8Array(buffer);
    expect(marshal.source({ id: 0xffffffff }, bytes, 0)).toBe(true);
    expect(bytes[0]).toBe(0xff);
    expect(bytes[1]).toBe(0xff);
    expect(bytes[2]).toBe(0xff);
    expect(bytes[3]).toBe(0xff);
  });

  it('marshals as bigendian', () => {
    const buffer = new ArrayBuffer(4);
    const bytes = new Uint8Array(buffer);
    expect(marshal.source({ id: 0xaabbccdd }, bytes, 0)).toBe(true);
    expect(bytes[0]).toBe(0xaa);
    expect(bytes[1]).toBe(0xbb);
    expect(bytes[2]).toBe(0xcc);
    expect(bytes[3]).toBe(0xdd);
  });

  it('marshals to an offset', () => {
    const buffer = new ArrayBuffer(5);
    const bytes = new Uint8Array(buffer);
    bytes[0] = 0xcd;
    expect(marshal.source({ id: 0xaabbccdd }, bytes, 1)).toBe(true);
    expect(bytes[0]).toBe(0xcd);
    expect(bytes[1]).toBe(0xaa);
    expect(bytes[2]).toBe(0xbb);
    expect(bytes[3]).toBe(0xcc);
    expect(bytes[4]).toBe(0xdd);
  });

  it('refuses to overrun buffer', () => {
    const buffer = new ArrayBuffer(4);
    const bytes = new Uint8Array(buffer);
    bytes[0] = 0xcd;
    bytes[1] = 0xcd;
    bytes[2] = 0xcd;
    bytes[3] = 0xcd;
    expect(marshal.source({ id: 0x00000000 }, bytes, 1)).toBe(false);
    expect(bytes[0]).toBe(0xcd);
    expect(bytes[1]).toBe(0xcd);
    expect(bytes[2]).toBe(0xcd);
    expect(bytes[3]).toBe(0xcd);
  });

});
