import { Address } from '../address';
import { AddressScope } from '../address_scope';
import { LogFunction, LogLevel } from '../log_function';
import { Reader } from '../reader';
import { Registry } from '../registry';
import { Segment } from '../segment';
import { Source } from '../source';
import { Writer } from '../writer';
import { AddressSet } from './address_set';
import { Binding } from './binding';
import { Channel } from './channel';
import { Discovery } from './discovery';
import { DiscoveryResult } from './discovery_result';
import { ResponseHandler } from './response_handler';
import { RetryLogic } from './retry_logic';

function isAddress(scope: AddressScope): scope is Address {
    return (scope as Address).namespace !== undefined;
}

export class Remote implements Registry {
  private readonly authRetry = new RetryLogic();
  private readonly hosts = new Map<string, RetryLogic>();
  private readonly connecting = new Map<string, AddressSet>();
  private readonly connected = new Map<string, AddressSet>();
  private readonly draining = new Map<string, AddressSet>();
  private readonly channels = new Map<string, Channel>();

  constructor(readonly disco: Discovery, private readonly log: LogFunction) {}

  public reader(addr: Address): Reader { return this.channel(addr); }
  public writer(addr: Address): Writer { return this.channel(addr); }

  public refreshAuth(): Promise<boolean> {
    const chain: Promise<boolean>[] = [];
    for (const [host, addrs] of this.connected) {
      chain.push(addrs.binding.refresh(this.disco.update(host)));
    }

    for (const [, addrs] of this.connecting) {
      addrs.refreshOnConnected = new ResponseHandler<boolean>();
      chain.push(addrs.refreshOnConnected.promise);
    }

    return Promise.all(chain).then((results) => {
        return !results.includes(false);
      });
  }

  public onSent(addr: Address, src: Source, at: Segment, content: ArrayBuffer): void {
    this.channel(addr).onSent(src, at, content);
  }

  public onLost(addr: Address, src: Source, at: Segment): void {
    this.channel(addr).onLost(src, at);
  }

  public onClosed(addr: Address, src: Source, endPosition: number): void {
    this.channel(addr).onClosed(src, endPosition);
  }

  public onHostExpiring(binding: Binding): void {
    this.log(LogLevel.Debug, 'expiring', binding.hostUrl);
    binding.refresh(this.disco.update(binding.hostUrl));
  }

  public onHostDraining(binding: Binding): void {
    this.log(LogLevel.Debug, 'draining', binding.hostUrl);
    let map = this.draining.get(binding.hostUrl);
    if (map) {
      for (const addr of map.addrs) {
        this.channel(addr).bind(null);
      }
    }

    map = this.connected.get(binding.hostUrl);
    if (map && map.binding === binding) {
      this.draining.set(binding.hostUrl, map);
      this.connected.delete(binding.hostUrl);
    } else {
      map = this.connecting.get(binding.hostUrl);
      if (map && map.binding === binding) {
        this.draining.set(binding.hostUrl, map);
        this.connecting.delete(binding.hostUrl);
      }
    }

    if (map) {
      const addrs = new Set<Address>(map.addrs);
      for (const addr of addrs) {
        this.requestHost(addr, true, map.binding);
      }
    }
  }

  public onHostClosed(binding: Binding): void {
    this.log(LogLevel.Debug, 'closed', binding.hostUrl);
    let map = this.connected.get(binding.hostUrl);
    if (map && map.binding === binding) {
      this.connected.delete(binding.hostUrl);
    } else {
      map = this.connecting.get(binding.hostUrl);
      if (map && map.binding === binding) {
        this.connecting.delete(binding.hostUrl);
      } else {
        map = this.draining.get(binding.hostUrl);
        if (map && map.binding === binding) {
          this.draining.delete(binding.hostUrl);
        }
      }
    }
    if (map) {
      for (const addr of map.addrs) {
        this.channel(addr).bind(null);
      }
    }
  }

  private channel(addr: Address): Channel {
    let channel = this.channels.get(addr.key);
    if (channel) {
      return channel;
    }
    const parents: Channel[] = [];
    for (const parent of addr.parents) {
      if (isAddress(parent)) {
        parents.push(this.channel(parent));
      }
    }
    channel = new Channel(addr, this.requestHost.bind(this), parents);
    this.channels.set(addr.key, channel);
    return channel;
  }

  private requestHost(addr: Address, connect: boolean, current: Binding): void {
    if (current) {
      let map = this.connected.get(current.hostUrl);
      if (!map || map.binding !== current) {
        map = this.connecting.get(current.hostUrl);
      }
      if (!map || map.binding !== current) {
        map = this.draining.get(current.hostUrl);
      }
      if (map && map.binding === current) {
        map.addrs.delete(addr);
        if (map.addrs.size === 0) {
          current.close();
        }
      }
    }
    if (connect) {
      this.findHost(addr, 0);
    } else {
      this.channel(addr).bind(null);
    }
  }

  private findHost(addr: Address, retryTime: number): void {
    let best: AddressSet | null = null;
    let binding: Binding | null = null;
    let score = -1;

    for (const [, map] of this.connected) {
      for (const scope of map.scopes) {
        if (scope.includes(addr) && scope.cardinality > score) {
          best = map;
          binding = map.binding;
          score = scope.cardinality;
        }
      }
    }
    for (const [, map] of this.connecting) {
      for (const scope of map.scopes) {
        if (scope.includes(addr) && scope.cardinality > score) {
          best = map;
          binding = null;
          score = scope.cardinality;
        }
      }
    }
    if (binding) {
      this.channel(addr).bind(binding);
    }
    if (best) {
      best.addrs.add(addr);
      return;
    }

    const remote = this;
    this.disco.find(addr).then(
      (result) => {
        let throttle = remote.hosts.get(result.hostUrl);
        if (!throttle) {
          throttle = new RetryLogic();
          remote.hosts.set(result.hostUrl, throttle);
        }
        // backoff retry per host here
        throttle.schedule(() => remote.onHostFound(addr, result));
      },
      () => {
        // todo: if the issue is auth, halt until refreshAuth()
        remote.authRetry.schedule(() => remote.findHost(addr, retryTime));
      },
    );
  }

  private onHostFound(addr: Address, result: DiscoveryResult): void {
    this.log(LogLevel.Info, 'found host', result.hostUrl);
    let binding: Binding | null = null;
    let map = this.connected.get(result.hostUrl);
    if (map) {
      binding = map.binding;
    } else {
      map = this.connecting.get(result.hostUrl);
    }

    if (map) {
      for (const scope of result.scopes) {
        map.scopes.add(scope);
      }
      map.addrs.add(addr);
      if (binding) {
        this.channel(addr).bind(binding);
      }
      return;
    }

    const created = new AddressSet(
      new Binding(
        result.hostUrl,
        this.onSent.bind(this),
        this.onLost.bind(this),
        this.onClosed.bind(this),
        this.onHostExpiring.bind(this),
        this.onHostClosed.bind(this),
        this.onHostDraining.bind(this),
        this.log,
      ),
      result.scopes,
    );
    this.connecting.set(result.hostUrl, created);

    created.addrs.add(addr);
    created.binding.initialize(result).then(
      () => {
        let toConnect = this.connecting.get(result.hostUrl);
        if (!toConnect || toConnect !== created) {
          created.binding.close();
        } else {
          this.connected.set(result.hostUrl, toConnect);
          this.connecting.delete(result.hostUrl);
          for (const toBind of toConnect.addrs) {
            this.channel(toBind).bind(toConnect.binding);
          }
          if (toConnect.refreshOnConnected) {
            const refresh = toConnect.binding.refresh(this.disco.update(result.hostUrl));
            toConnect.refreshOnConnected.bind(refresh);
            toConnect.refreshOnConnected = undefined;
          }
        }
      },
      () => { created.binding.close(); },
    );
  }
}
