import { Address } from '../address';
import * as errors from '../errors';
import { Segment } from '../segment';
import * as unmarshal from './unmarshal';

describe('unmarshal.address', () => {
    it('should unmarshal a valid address', () => {
        const addr = Address.parse('test@1?filter=value');
        const bytes = new Uint8Array(Buffer.from(addr.key));
        expect(unmarshal.address(bytes, 0)).toStrictEqual(addr);
    });

    it('should refuse an invalid address', () => {
        const bytes = new Uint8Array(Buffer.from('malformed'));
        expect(() => unmarshal.address(bytes, 0)).toThrow(errors.MissingVersion);
    });

    it('should unmarshal from an offset', () => {
        const addr = Address.parse('test@1?filter=value');
        const bytes = new Uint8Array(Buffer.concat([new Uint8Array([0, 0]), Buffer.from(addr.key)]));
        expect(unmarshal.address(bytes, 2)).toStrictEqual(addr);
    });
});

describe('unmarshal.position', () => {
    it('should unmarshal min value (0x00000000)', () => {
        const bytes = new Uint8Array([0, 0, 0, 0]);
        expect(unmarshal.position(bytes, 0)).toBe(0);
    });

    it('should unmarshal max value (0xfffffffff)', () => {
      const bytes = new Uint8Array([0xff, 0xff, 0xff, 0xff]);
      expect(unmarshal.position(bytes, 0)).toBe(0xffffffff);
    });

    it('should unmarshal big endian', () => {
        const bytes = new Uint8Array([0xaa, 0xbb, 0xcc, 0xdd]);
        expect(unmarshal.position(bytes, 0)).toBe(0xaabbccdd);
    });

    it('should unmarshal from an offset', () => {
      const bytes = new Uint8Array([0xcd, 0xaa, 0xbb, 0xcc, 0xdd, 0xcd]);
      expect(unmarshal.position(bytes, 1)).toBe(0xaabbccdd);
    });

    it('should not overrun the buffer', () => {
      const bytes = new Uint8Array([0xcd, 0xaa, 0xbb]);
      expect(unmarshal.position(bytes, 1)).toBeUndefined();
    });
});

describe('unmarshal.source', () => {
  it('should unmarshal min value (0x00000000)', () => {
    const bytes = new Uint8Array([0, 0, 0, 0]);
    expect(unmarshal.source(bytes, 0)).toEqual({ id: 0 });
  });

  it('should unmarshal max value (0xfffffffff)', () => {
    const bytes = new Uint8Array([0xff, 0xff, 0xff, 0xff]);
    expect(unmarshal.source(bytes, 0)).toEqual({ id: 0xffffffff });
  });

  it('should unmarshal big endian', () => {
    const bytes = new Uint8Array([0xaa, 0xbb, 0xcc, 0xdd]);
    expect(unmarshal.source(bytes, 0)).toEqual({ id: 0xaabbccdd });
  });

  it('should unmarshal from an offset', () => {
    const bytes = new Uint8Array([0xcd, 0xaa, 0xbb, 0xcc, 0xdd, 0xcd]);
    expect(unmarshal.source(bytes, 1)).toEqual({ id: 0xaabbccdd });
  });

  it('should not overrun the buffer', () => {
    const bytes = new Uint8Array([0xcd, 0xaa, 0xbb]);
    expect(unmarshal.source(bytes, 1)).toBeUndefined();
  });
});

describe('unmarshal.segment', () => {
  it('should unmarshal min value (0x00000000,0x00000000)', () => {
    const bytes = new Uint8Array([0, 0, 0, 0, 0, 0, 0, 0]);
    expect(unmarshal.segment(bytes, 0)).toStrictEqual(new Segment(0, 0));
  });

  it('should unmarshal max value (0xfffffffff, 0xfffffffff)', () => {
    const bytes = new Uint8Array([0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff]);
    expect(unmarshal.segment(bytes, 0)).toStrictEqual(new Segment(0xffffffff, 0xffffffff));
  });

  it('should unmarshal big endian', () => {
    const bytes = new Uint8Array([0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff, 0x11, 0x22]);
    expect(unmarshal.segment(bytes, 0)).toStrictEqual(new Segment(0xaabbccdd, 0xeeff1122));
  });

  it('should unmarshal from an offset', () => {
    const bytes = new Uint8Array([0xcd, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff, 0x11, 0x22]);
    expect(unmarshal.segment(bytes, 1)).toStrictEqual(new Segment(0xaabbccdd, 0xeeff1122));
  });

  it('should not overrun the buffer', () => {
    const bytes = new Uint8Array([0xcd, 0xaa, 0xbb, 0xcc, 0xdd]);
    expect(unmarshal.segment(bytes, 1)).toBeUndefined();
  });
});
