import * as utf8 from './utf8';

describe('utf8.byteLength', () => {
    it('should return the length of an ascii string', () => {
        expect(utf8.byteLength('hello')).toBe(5);
    });
});

describe('utf8.end', () => {
    it('should report the position of the next \0 for string termination', () => {
        const bytes = new Uint8Array(Buffer.from('hello\0hello'));
        expect(utf8.end(bytes, 0)).toBe(5);
    });

    it('should report the position of the next \0 for string termination from an offset', () => {
      const bytes = new Uint8Array(Buffer.from('hello\0hello\0hello'));
      expect(utf8.end(bytes, 6)).toBe(11);
    });

    it('should return undefined if no \0 is present', () => {
      const bytes = new Uint8Array(Buffer.from('hello'));
      expect(utf8.end(bytes, 6)).toBeUndefined();
    });
});

describe('utf8.marshal', () => {
    it('should copy ascii strings into a buffer', () => {
        const value = 'string';
        const bytes = new Uint8Array(value.length);
        expect(utf8.marshal(value, bytes, 0)).toBe(true);
        expect(bytes).toEqual(new Uint8Array(Buffer.from(value)));
    });

    it('should copy at the desired offset', () => {
        const value = 'string';
        const bytes = new Uint8Array(value.length + 1);
        bytes[0] = 0;
        expect(utf8.marshal(value, bytes, 1)).toBe(true);
        expect(bytes).toEqual(new Uint8Array(Buffer.from('\0' + value)));
    });

    it('refuse to overrun the buffer', () => {
      const value = 'string';
      const bytes = new Uint8Array(value.length - 1);
      expect(utf8.marshal(value, bytes, 0)).toBe(false);
    });
});

describe('utf8.unmarshal', () => {
  it('should extract null-terminated ascii strings from a buffer', () => {
    const bytes = new Uint8Array(Buffer.from('string\0hello'));
    expect(utf8.unmarshal(bytes, 0)).toBe('string');
  });

  it('should extract the entire string if not null-terminated', () => {
    const bytes = new Uint8Array(Buffer.from('string'));
    expect(utf8.unmarshal(bytes, 0)).toBe('string');
  });

  it('should return undefined if buffer is undersized', () => {
    const bytes = new Uint8Array(Buffer.from('string'));
    expect(utf8.unmarshal(bytes, 20)).toBeUndefined();
  });
});
