// tslint:disable-next-line: no-submodule-imports
import 'source-map-support/register';

import { App, Tag } from '@aws-cdk/core';
import { EcrRepositoryStack } from './stacks/ecr-repository';
import { ECSClusterStack } from './stacks/ecs-cluster';
import { ECSFargateStack } from './stacks/ecs-fargate';
import { S3BucketStack } from './stacks/s3-bucket';
import { SecretsStack } from './stacks/secrets';
import { VpcStack } from './stacks/vpc';

const DYNAMICCONF_S3BUCKET = "eml-loadtest";

const props = {
  env: {
    account: '947140302170', // twitch-devhub-testing
    region: 'us-west-2',
  }
};

const app = new App();
Tag.add(app, 'project', 'eml-loadtest');

const emlLoadtestVpc = new VpcStack(app, 'EMLLoadtestVPC', props);

const emlLoadtestEcrRepository = new EcrRepositoryStack(app, 'EMLLoadtestECRRepository', props);
// After initial deploy: the registry needs to have a container image,
// make sure to push an image before deploying dependent infra.

const secrets = new SecretsStack(app, 'EMLLoadtestDatadogSecret', props);
// After initial deploy: Copy the API-Key secret from https://app.datadoghq.com/account/settings#api
// and pase into AWS console > Secrets Manager > datadog-api-key > Secret Value
// Copy the greeter-client-auth secret from twitch-eml-dev into dev-greeter-client-s2s
// Copy the greeter-client-auth secret from twitch-eml-prod into prod-greeter-client-s2s

const s3Bucket = new S3BucketStack(app, 'EMLLoadtestS3Bucket', {...props, bucketName: DYNAMICCONF_S3BUCKET});
// After initial deploy: add all the TOML files with dynamic configuration for each cluster

// ECS Fargate services with different DynamicConfigs.
const ecsCluster = new ECSClusterStack(app, 'EMLLoadtestECSCluster', {...props, vpc: emlLoadtestVpc.vpc});
[
  { testInstance: "dev01" },
  { testInstance: "dev02" },
  { testInstance: "dev03" },
  { testInstance: "dev04" },
].forEach(svc => {
  new ECSFargateStack(app, `EMLLoadtest${svc.testInstance}`, {
    ...props,
    vpc: emlLoadtestVpc.vpc,
    cluster: ecsCluster.cluster,
    ecrRepo: emlLoadtestEcrRepository.ecrRepo,
    s3Bucket: s3Bucket.bucket,
    datadogApiKey: secrets.datadogApiKey,
    testInstance: svc.testInstance,
    greeterClientAuth: secrets.devGreeterClientS2s,
  });
})
