import ec2 = require('@aws-cdk/aws-ec2');
import ecr = require('@aws-cdk/aws-ecr');
import ecs = require('@aws-cdk/aws-ecs');
import s3 = require('@aws-cdk/aws-s3');
import secretsmanager = require('@aws-cdk/aws-secretsmanager');
import cdk = require('@aws-cdk/core');

interface ECSFargateStackProps extends cdk.StackProps {
    vpc: ec2.IVpc;
    cluster: ecs.ICluster;
    ecrRepo: ecr.IRepository;
    s3Bucket: s3.IBucket; // to read dynamic config file
    datadogApiKey: secretsmanager.ISecret;
    greeterClientAuth: secretsmanager.ISecret; // for authorization

    testInstance: string; // "dev01", "dev02", etc., identifies the Fargate Service, and DynamicConf file in S3
}

export class ECSFargateStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: ECSFargateStackProps) {
    super(scope, id, props);

    const serviceName = `eml_loadtest_${props.testInstance}`;
    const dynamicConfS3File = `dynamic_config/${props.testInstance}.toml`; // e.g. "dynamic_config/dev01.toml"

    const taskDefinition = new ecs.FargateTaskDefinition(this, 'TaskDefinition', {
      cpu: 4096, // 4 vCPU
      memoryLimitMiB: 8192, // 8 GB (minimum available for the cpu value)
    });

    props.s3Bucket.grantRead(taskDefinition.taskRole); // allow to read dynamic config file form the S3 bucket

    // eml_loadtest container
    const container = taskDefinition.addContainer('TaskContainer', {
        image: ecs.ContainerImage.fromEcrRepository(props.ecrRepo, 'latest'), // Requires props.ecrRepo to exist and have a registered docker image
        logging: new ecs.AwsLogDriver({ streamPrefix: 'eml_loadtest' }),
        dockerLabels: {
          'com.docker.compose.service': 'eml_loadtest', // used by datadog autodiscovery DD_DOCKER_LABELS_AS_TAGS
        },
        environment: {
          TARGETENV: "dev",
          TESTINSTANCE: props.testInstance,
          DYNAMICCONFFILE: dynamicConfS3File,
          DYNAMICCONFS3BUCKET: props.s3Bucket.bucketName,
        },
        secrets: {
          GREETER_CLIENT_AUTH: ecs.Secret.fromSecretsManager(props.greeterClientAuth),
        },
    });
    container.addUlimits({
      name: ecs.UlimitName.NOFILE, // number of open files
      softLimit: 10240,
      hardLimit: 10240,
    });

    // Datadog agent sidecar container: https://github.com/DataDog/integrations-core/tree/master/ecs_fargate
    const datadogContainer = taskDefinition.addContainer('DatadogAgent', {
      cpu: 10,
      memoryReservationMiB: 256,
      environment: {
        ECS_FARGATE: 'true',
        DD_DOCKER_LABELS_AS_TAGS: '{"com.docker.compose.service":"service_name"}', // "{"<LABEL_NAME_TO_COLLECT>":"<TAG_KEY_FOR_DATADOG>"}"
        DD_DOGSTATSD_NON_LOCAL_TRAFFIC: 'true', // send custom metrics by listening to DogStatsD packets from other containers
        DD_PROCESS_AGENT_ENABLED: 'true', // because Amazon controls the underlying hosts for Fargate, live processes can only be collected from the Datadog Agent container
      },
      image: ecs.ContainerImage.fromRegistry('datadog/agent:latest'),
      essential: false, // stop when the test run is done
      logging: new ecs.AwsLogDriver({ streamPrefix: 'DataDog' }),
      secrets: {
        DD_API_KEY: ecs.Secret.fromSecretsManager(props.datadogApiKey),
        GREETER_CLIENT_AUTH: ecs.Secret.fromSecretsManager(props.greeterClientAuth),
      },
    });
    datadogContainer.addPortMappings({ containerPort: 8125 });

    // Service
    new ecs.FargateService(this, 'TaskService', {
      serviceName,
      cluster: props.cluster,
      taskDefinition,
      assignPublicIp: false,
      desiredCount: 0, // managed later with a shell script to start loadtest runs
    });
  }
}
