package config

import (
	"io"
	"io/ioutil"
	"os"
	"reflect"
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"

	toml "github.com/pelletier/go-toml"
)

const (
	checkUpdatesInterval = 5 * time.Second
)

type DynamicConfig struct {
	Writers   int // number of writers
	Listeners int // number of listeners
	Addrs     int // number of addresses to distribute writers/listeners
	Offset    int // initial address number to be used on the filter, to spread separate from other running tests

	MsgFreqMin int  // min msg/minute per writer
	MsgFreqMax int  // max msg/minute per writer
	MsgSizeMin int  // min msg size
	MsgSizeMax int  // max msg size
	MsgIsDelta bool // send messages as deltas

	LogLevel    string // trace|debug|info|warning|error
	LogLevelReg string // same as LogLevel but used for library registry code only
}

type DynamicConfigMngr struct {
	s3Bucket string
	filePath string

	s3Cli *s3.S3

	loadNextAt time.Time
	loading    bool
	loadedLast *DynamicConfig

	loadResultVal *DynamicConfig
	loadResultErr error
	loadResultMux sync.Mutex
}

func NewDynamicConfigMngr(s3Bucket string, filePath string) (*DynamicConfigMngr, error) {
	m := &DynamicConfigMngr{
		s3Bucket: s3Bucket,
		filePath: filePath,
	}
	if s3Bucket != "" {
		sess, err := session.NewSession(&aws.Config{Region: aws.String("us-west-2")}) // reads credentials from environment variables or .aws/credentials file
		if err != nil {
			return nil, err
		}
		m.s3Cli = s3.New(sess)
	}
	return m, nil
}

// TickAndCheckUpdatesAsync should be called from the main loop at quick intervals with the current time (now).
// Returns the new loaded configuration if there are changes since the last time it was loaded.
func (m *DynamicConfigMngr) TickAndCheckUpdatesAsync(now time.Time) (*DynamicConfig, error) {
	if now.Before(m.loadNextAt) {
		return nil, nil // last load was too recent
	}

	if !m.loading {
		m.loading = true
		go func() { // load async
			loaded, loadErr := m.Load()
			m.SetLoadResult(loaded, loadErr)
		}()
		return nil, nil // loading ...
	}

	loaded, loadErr := m.GetLoadResult()
	if loaded == nil && loadErr == nil {
		return nil, nil // still loading ...
	}

	m.SetLoadResult(nil, nil)
	m.loading = false
	m.loadNextAt = now.Add(checkUpdatesInterval) // new interval to check for updates after loaded (the time to load was added to the real interval)

	if loadErr != nil {
		return nil, loadErr
	}
	if reflect.DeepEqual(loaded, m.loadedLast) { // shallow equality check
		return nil, nil // nothing changed since last check
	}
	m.loadedLast = loaded
	return loaded, nil
}

func (m *DynamicConfigMngr) SetLoadResult(dc *DynamicConfig, err error) {
	m.loadResultMux.Lock()
	defer m.loadResultMux.Unlock()
	m.loadResultVal, m.loadResultErr = dc, err
}

func (m *DynamicConfigMngr) GetLoadResult() (*DynamicConfig, error) {
	m.loadResultMux.Lock()
	defer m.loadResultMux.Unlock()
	return m.loadResultVal, m.loadResultErr
}

func (m *DynamicConfigMngr) Load() (*DynamicConfig, error) {
	if m.s3Bucket != "" {
		return m.LoadFromS3()
	} else {
		return m.LoadFromFile()
	}
}

func (m *DynamicConfigMngr) LoadFromS3() (*DynamicConfig, error) {
	result, err := m.s3Cli.GetObject(&s3.GetObjectInput{
		Bucket: aws.String(m.s3Bucket),
		Key:    aws.String(m.filePath),
	})
	if err != nil {
		return nil, err
	}
	defer result.Body.Close()
	return ParseToml(result.Body)
}

func (m *DynamicConfigMngr) LoadFromFile() (*DynamicConfig, error) {
	file, err := os.Open(m.filePath)
	if err != nil {
		return nil, err
	}
	defer file.Close()
	return ParseToml(file)
}

func ParseToml(body io.Reader) (*DynamicConfig, error) {
	bodyBytes, err := ioutil.ReadAll(body)
	if err != nil {
		return nil, err
	}
	dc := &DynamicConfig{}
	err = toml.Unmarshal(bodyBytes, dc)
	return dc, err
}

func HasChanges(dc1, dc2 *DynamicConfig) bool {
	return true
}
