package config

import (
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/alexflint/go-arg"
)

// ParseFlagsEnvConfig reads program args and environment variables
func ParseFlagsEnvConfig() FlagsEnvConfig {
	conf := FlagsEnvConfigDefaults()
	arg.MustParse(&conf)
	return conf
}

type FlagsEnvConfig struct {
	TargetENV         string             `arg:"env,-t" help:"Target EML environment to test [local,dev,prod]. It pre-defines the URL for the registry threshold to connect and is used in metrics."`
	TestInstance      string             `arg:"env" help:"Name to identify the fargate service and DynamicConf (should match). E.g. dev01, dev02, etc."`
	Insecure          bool               `arg:"env,-i" help:"InsecureSkipVerify when getting the TLS certificates"`
	GreeterClientAuth stream.OpaqueBytes `arg:"--greeter-client-auth,-a,env:GREETER_CLIENT_AUTH" help:"S2S secret for connections; uses anonymous requests if not set"`

	DynamicConfFile     string `arg:"env,-f" help:"Path for the TOML file with dynamic configuration. Used as object key on the S3 bucket, or as local file path if S3 bucket is not specified."`
	DynamicConfS3Bucket string `arg:"env,-b" help:"S3 bucket name for the remote TOML file with dynamic configuration. If not specified, the file is loaded from a local file path."`

	LimitAddedPerTick int `arg:"env,-l" help:"Number of writers and listeners that are allocated per main-loop tick."`
	LimitConnecting   int `arg:"env,-c" help:"Number of concurrent workers to handle connecting/disconnecting writers and listeners in the background."`
}

func (c FlagsEnvConfig) Description() string {
	return "eml-loadtest generates fake writers and listeners for the EML service, using a dynamic config file locally or in a S3 bucket to update the test run in-flight."
}

func FlagsEnvConfigDefaults() FlagsEnvConfig {
	return FlagsEnvConfig{
		TargetENV: "local",
		Insecure:  false,

		TestInstance:        "local",
		DynamicConfFile:     "config/dynamic_config/local.toml",
		DynamicConfS3Bucket: "", // from local file by default

		LimitAddedPerTick: 10,
		LimitConnecting:   4,
	}
}

// TargetEnvConf expands the TargetENV into its values. Panics if TargetEnv is invalid.
func (c FlagsEnvConfig) TargetENVConf() TargetENVConf {
	switch c.TargetENV {
	case "local":
		return TargetENVConf{AWSName: "localhost", AWSID: "local", URL: "wss://localhost:3002", UseDatadog: false, TestInstance: c.TestInstance}
	case "dev":
		return TargetENVConf{AWSName: "twitch-eml-dev", AWSID: "565915620853", URL: "wss://connect.dev.eml.twitch.a2z.com/", UseDatadog: true, TestInstance: c.TestInstance}
	case "prod":
		return TargetENVConf{AWSName: "twitch-eml-prod", AWSID: "342135511598", URL: "wss://connect.prod.eml.twitch.a2z.com/", UseDatadog: true, TestInstance: c.TestInstance}
	default:
		panic("Invalid config TargetENV: " + c.TargetENV)
	}
}

// Target envs are configured statically (they don't change much).
type TargetENVConf struct {
	AWSName      string // account name
	AWSID        string // numeric account id
	TestInstance string // dev01, dev02, etc
	URL          string // URL to threshold on that account
	UseDatadog   bool   // logging strategy
}

// GetTags implements metrics.TagSource interface, required by datadog.NewTracker
func (t TargetENVConf) GetTags() []string {
	return []string{
		"service_name:eml-loadtest",
		"aws_account:" + t.AWSID,
		"test_instance:" + t.TestInstance,
	}
}
