package emlclient

import (
	"fmt"
	"sync/atomic"

	"code.justin.tv/devhub/e2ml/libs/logging"
)

type DynamicLogger struct {
	name  string
	level int32 // accessed through atomic operations for better performance
}

func NewDynamicLogger(name string, level logging.Level) *DynamicLogger {
	return &DynamicLogger{name: name, level: int32(level)}
}

func (l *DynamicLogger) Log(level logging.Level, msg ...interface{}) {
	if l.GetLevel() <= level {
		fmt.Println(msg...)
	}
}

func (l *DynamicLogger) GetLevel() logging.Level {
	return logging.Level(atomic.LoadInt32(&l.level))
}

func (l *DynamicLogger) SetLevel(level logging.Level) {
	atomic.StoreInt32(&l.level, int32(level))
}

func (l *DynamicLogger) UpdateLevelStr(lvlStr string) {
	prevLvlStr := l.GetLevel().String()
	if lvlStr == prevLvlStr {
		return // nothing to update
	}

	level, ok := logging.ParseLevel(lvlStr)
	if !ok {
		l.Log(logging.Error, fmt.Sprintf("Logger[%s] Update Level from %q to %q. ERROR: new level is invalid, keep using previous level", l.name, prevLvlStr, lvlStr))
		return // ignore
	}
	l.SetLevel(level)
	l.Log(logging.Info, fmt.Sprintf("Logger[%s] Update Level from %q to %q", l.name, prevLvlStr, lvlStr))
}
