package emlclient

import (
	"fmt"
	"sync"
	"time"

	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

type lstats struct {
	msgRead            metrics.Count
	msgRepeated        metrics.Count
	msgLost            metrics.Count
	msgLostAndRepeated metrics.Count
	totalSampled       metrics.Count
	totalElapsed       metrics.Count
}

func newListenerStats(stats metrics.Tracker) lstats {
	return lstats{
		msgRead:            stats.Count("listener.data_received", []string{"dropped:false", "duplicate:false"}),
		msgRepeated:        stats.Count("listener.data_received", []string{"dropped:false", "duplicate:true"}),
		msgLost:            stats.Count("listener.data_received", []string{"dropped:true", "duplicate:false"}),
		msgLostAndRepeated: stats.Count("listener.data_received", []string{"dropped:true", "duplicate:true"}),
		totalSampled:       stats.Count("listener.total_sampled", []string{}),
		totalElapsed:       stats.Count("listener.total_elapsed", []string{}),
	}
}

type Listener struct {
	id         int // also index in the list of concurrent listeners in this process
	logPrefix  string
	msgFactory *MessageFactory
	reg        stream.Registry

	reader  stream.Reader
	tracker *stream.Tracker

	mutex  sync.Mutex
	stats  lstats
	logger logging.Function
}

var _ Task = (*Listener)(nil)

func NewListener(id int, reg stream.Registry, msgFactory *MessageFactory, logger logging.Function, stats metrics.Tracker) *Listener {
	return &Listener{
		stats:      newListenerStats(stats),
		id:         id,
		logPrefix:  fmt.Sprintf("Listener#%d: ", id),
		msgFactory: msgFactory,
		reg:        reg,
		tracker:    new(stream.Tracker),
		logger:     logger,
	}
}

func (l *Listener) ID() int      { return l.id }
func (l *Listener) Type() string { return "listener" }

// Start attempts to listen to the given address, potentially leaving any
// previously set address.
func (l *Listener) Start(addr stream.Address) bool {
	now := time.Now()
	key := addr.Key()
	l.mutex.Lock()
	defer l.mutex.Unlock()

	// NO-OP shortcut
	if l.reader != nil && key == l.reader.Address().Key() {
		return true
	}

	// only leave previous address after joining new one to avoid bouncing connection
	prev := l.reader

	l.reader = l.reg.Reader(addr)
	_, err := l.reader.Join(l).Result()
	if err != nil {
		l.reader = nil // don't store dead readers
	}

	if prev != nil {
		_, lerr := prev.Leave(l).Result()
		err = lifecycle.CombineErrors(err, lerr)
	}

	if err != nil {
		l.log(logging.Debug, "start error: {Addr:%v, Elapsed:%v, Err:%v}", addr, time.Since(now), err)
	} else {
		l.log(logging.Trace, "start: {Addr:%v, Elapsed:%v}", addr, time.Since(now))
	}
	return l.reader != nil
}

// Stop shuts down any active listener, which will automatically disconnect the registry
func (l *Listener) Stop() bool {
	now := time.Now()
	l.mutex.Lock()
	defer l.mutex.Unlock()

	// NO-OP shortcut
	if l.reader == nil {
		return true
	}

	addr := l.reader.Address()
	success, err := l.reader.Leave(l).Result()
	if success {
		l.reader = nil
	}

	if err != nil {
		l.log(logging.Debug, "stop error: {Addr:%v, Elapsed:%v, Err:%v}", addr, time.Since(now), err)
	} else {
		l.log(logging.Trace, "stop: {Addr:%v, Elapsed:%v}", addr, time.Since(now))
	}
	return success
}

//
// stream.Listener interface implementation
//

func (l *Listener) Current(addr stream.Address) (stream.SourceID, stream.Position) {
	return l.tracker.Current()
}

func (l *Listener) OnDataReceived(msg stream.Message) bool {
	if l.tracker.Set(msg.Source(), msg.At().End) {
		l.stats.msgRead.Add(1)
		if elapsed := l.getElapsed(msg.Data()); elapsed > 0 {
			l.stats.totalSampled.Add(1)
			l.stats.totalElapsed.Add(int64(elapsed))
			l.log(logging.Trace, "Data received {Addr:%v, SourceID:%X, Segment:[%d:%d], DataLength: %d, Elapsed: %v}", msg.Address(), msg.Source(), msg.At().Start, msg.At().End, len(msg.Data()), elapsed)
		} else {
			l.log(logging.Trace, "Data received {Addr:%v, SourceID:%X, Segment:[%d:%d], DataLength: %d}", msg.Address(), msg.Source(), msg.At().Start, msg.At().End, len(msg.Data()))
		}
	} else {
		l.stats.msgRepeated.Add(1)
		l.log(logging.Debug, "Data received with previous tracker segment {Addr:%v, SourceID:%X, Segment:[%d:%d], DataLength: %d}", msg.Address(), msg.Source(), msg.At().Start, msg.At().End, len(msg.Data()))
	}
	return true
}

func (l *Listener) OnDataLost(msg stream.MessageDescription) bool {
	if l.tracker.Set(msg.Source(), msg.At().End) {
		l.stats.msgLost.Add(1)
		l.log(logging.Trace, "Data lost {Addr:%v, SourceID:%X, Segment:[%d:%d]}", msg.Address(), msg.Source(), msg.At().Start, msg.At().End)
	} else {
		l.stats.msgLostAndRepeated.Add(1)
		l.log(logging.Debug, "Data lost detected with previous tracker segment {Addr:%v, SourceID:%X, Segment:[%d:%d]}", msg.Address(), msg.Source(), msg.At().Start, msg.At().End)
	}
	return true
}

func (l *Listener) OnStreamClosed(addr stream.Address, err error) bool {
	if err != nil {
		l.log(logging.Debug, "OnStreamClosed {Addr:%v, Err:%v}", addr, err)
	} else {
		l.log(logging.Trace, "OnStreamClosed {Addr:%v}", addr)
	}
	return true
}

//
// Helpers
//

func (l *Listener) log(lvl logging.Level, msg string, args ...interface{}) {
	l.logger(lvl, fmt.Sprintf(l.logPrefix+msg, args...))
}

// getElapsed returns the time duration since the message was created if and only if
// the message was created by a MessageFactory using the same prefix. Otherwise, it
// returns duration 0
func (l *Listener) getElapsed(data []byte) time.Duration {
	timeSent, _ := l.msgFactory.ParseTime(data)
	return timeSince(timeSent)
}
