package emlclient

import (
	"bytes"
	"encoding/binary"
	"math/rand"
	"time"

	"code.justin.tv/devhub/e2ml-loadtest/config"
)

// MessageFactory generates a list of randomized messages that use a timestamp and prefix to allow round trip time tracking
type MessageFactory struct {
	minSize   int
	maxSize   int
	minFreq   int
	maxFreq   int
	isDelta   bool
	prefix    []byte
	prefixLen int
}

// NewMessageFactory creates a message factory with sanity checked message sizes
func NewMessageFactory(minSize, maxSize int, prefix []byte) *MessageFactory {
	prefixLen := len(prefix)
	factory := &MessageFactory{prefix: prefix, prefixLen: prefixLen}
	factory.updateSizes(minSize, maxSize)
	return factory
}

// Update configures the message factory to match the dynamic confguration settings
func (m *MessageFactory) Update(dynamicConfig *config.DynamicConfig) {
	m.updateSizes(dynamicConfig.MsgSizeMin, dynamicConfig.MsgSizeMax)
	m.minFreq = dynamicConfig.MsgFreqMin
	m.maxFreq = dynamicConfig.MsgFreqMax
	m.isDelta = dynamicConfig.MsgIsDelta
}

// Next builds and returns a message along with the delay to use before sending it
// TODO : consider caching pre-generated message listing
func (m *MessageFactory) Next() ([]byte, bool, time.Duration) {
	size := m.nextSize()
	delay := timeToNextMsg(m.minFreq, m.maxFreq)
	msg := make([]byte, size)
	binary.BigEndian.PutUint64(msg, uint64(time.Now().UnixNano()))
	copy(msg[8:], m.prefix)
	if size > 8+m.prefixLen {
		rand.Read(msg[8+m.prefixLen:])
	}
	return msg, m.isDelta, delay
}

// ParseTime extracts the time from a message if it has the same prefix as the current MessageFactory
func (m *MessageFactory) ParseTime(msg []byte) (time.Time, bool) {
	if len(msg) < m.prefixLen+8 || !bytes.Equal(m.prefix, msg[8:8+m.prefixLen]) {
		return time.Time{}, false
	}
	return time.Unix(0, int64(binary.BigEndian.Uint64(msg))), true
}

func (m *MessageFactory) nextSize() int {
	size := m.minSize
	if size < m.maxSize {
		size += rand.Intn(m.maxSize - size + 1)
	}
	return size
}

func (m *MessageFactory) updateSizes(minSize, maxSize int) {
	if minSize < 8+m.prefixLen {
		minSize = 8 + m.prefixLen
	}
	if maxSize < minSize {
		maxSize = minSize + m.prefixLen
	}
	m.minSize = minSize
	m.maxSize = maxSize
}

func timeToNextMsg(msgFreqMin int, msgFreqMax int) time.Duration {
	msgsPerMinute := randBetween(msgFreqMin, msgFreqMax)
	sleepMinutes := float64(1) / float64(msgsPerMinute)
	return time.Duration(sleepMinutes * float64(time.Minute))
}

func randBetween(min, max int) int {
	return min + rand.Intn(max-min+1)
}
