package emlclient

import (
	"fmt"
	"sync"
	"time"

	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

type Writer struct {
	id         int // also index in the list of concurrent writers in this process
	logPrefix  string
	msgFactory *MessageFactory
	reg        stream.Registry

	writer        stream.Writer
	sendNextMsgAt time.Time

	mutex  sync.Mutex
	logger logging.Function
}

var _ Task = (*Writer)(nil)

func NewWriter(id int, reg stream.Registry, msgFactory *MessageFactory, logger logging.Function) *Writer {
	return &Writer{
		id:         id,
		logPrefix:  fmt.Sprintf("Writer#%d: ", id),
		msgFactory: msgFactory,
		reg:        reg,
		logger:     logger,
	}
}

func (w *Writer) ID() int      { return w.id }
func (w *Writer) Type() string { return "writer" }

// Start opens up a writer connection; messages will be sent on tick
func (w *Writer) Start(addr stream.Address) bool {
	now := time.Now()
	key := addr.Key()
	w.mutex.Lock()
	defer w.mutex.Unlock()

	// NO-OP shortcut
	if w.writer != nil && key == w.writer.Address().Key() {
		return true
	}

	success := w.stopInternal() // close the previous writer if applicable

	w.writer = w.reg.Writer(addr) // we won't know about errors until first send

	w.log(logging.Trace, "start: {Addr:%v, Elapsed:%v}", addr, time.Since(now))

	return success
}

// Stop closes a writer connection
func (w *Writer) Stop() bool {
	w.mutex.Lock()
	defer w.mutex.Unlock()

	return w.stopInternal()
}

// Note: lock is acquired in wrappers around this function
func (w *Writer) stopInternal() bool {
	now := time.Now()
	// NO-OP shortcut
	if w.writer == nil {
		return true
	}

	addr := w.writer.Address()

	if err := w.writer.Close(); err != nil {
		w.log(logging.Debug, "stop error: {Addr:%v, Elapsed:%v, Err:%v}", addr, timeSince(now), err)
	} else {
		w.log(logging.Trace, "stop: {Addr:%v, Elapsed:%v}", addr, timeSince(now))
	}
	return true
}

// Tick sends messages at stable frequecy intervals. It should be called from the main loop
// to ensure that the messages are sent as close to the intended frequency as possible.
func (w *Writer) Tick(now time.Time) {
	w.mutex.Lock()
	writer := w.writer // writer can be updated from background thread, everything else is foreground only
	w.mutex.Unlock()

	if writer == nil || now.Before(w.sendNextMsgAt) {
		return
	}

	// it's time and we have a writer: send the message and queue the next one
	msg, isDelta, delay := w.msgFactory.Next()
	last := w.sendNextMsgAt
	w.sendNextMsgAt = now.Add(delay)

	addr := writer.Address()
	length := len(msg)

	go func() { // handle result asynchronously
		promise := writer.Send(msg, isDelta)
		if _, err := promise.Result(); err != nil {
			w.log(logging.Debug, "Write error: {Addr:%v, MsgLen:%d, Elapsed:%v, Err:%v}", addr, length, timeSince(now), err)
		} else {
			w.log(logging.Trace, "Message sent {Addr:%v, MsgLen:%d, Elapsed:%v, TimeSinceLast:%s}", addr, length, timeSince(now), timeSince(last))
		}
	}()
}

func (w *Writer) log(lvl logging.Level, msg string, args ...interface{}) {
	w.logger(lvl, fmt.Sprintf(w.logPrefix+msg, args...))
}

//
// Helpers
//

func timeSince(t time.Time) time.Duration {
	if t.IsZero() {
		return time.Duration(0)
	}
	return time.Since(t)
}
