package errors

// DictionaryBuilder is used to construct an error builder; this separation of responsibility
// allows the resulting ErrorDictionary to be threadsafe by making it read-only. The builder is not
// threadsafe.
type DictionaryBuilder interface {
	Include(...ErrorCodeError) DictionaryBuilder                       // for error types without details
	Map(errorCode string, using func(Details) error) DictionaryBuilder // for error types with details
	IncludeAll(Dictionary) DictionaryBuilder                           // allow combining other dictionaries
	Build() Dictionary
}

// NewDictionaryBuilder constructs a new DictionaryBuilder
func NewDictionaryBuilder() DictionaryBuilder {
	return &dictionaryBuilder{make(dictionary)}
}

type dictionaryBuilder struct{ values dictionary }

func (d *dictionaryBuilder) Build() Dictionary { return d.values.copy() }

func (d *dictionaryBuilder) Map(code string, using func(Details) error) DictionaryBuilder {
	d.values[code] = using
	return d
}

func (d *dictionaryBuilder) Include(errs ...ErrorCodeError) DictionaryBuilder {
	for _, err := range errs {
		errx := err
		d.values[err.ErrorCode()] = func(Details) error { return errx }
	}
	return d
}

func (d *dictionaryBuilder) IncludeAll(dict Dictionary) DictionaryBuilder {
	if cast, ok := dict.(dictionary); ok {
		for k, v := range cast {
			d.values[k] = v
		}
	}
	return d
}
