package logging

import (
	"bufio"
	"errors"
	"fmt"
	"io"
	"os"
	"sync"
	"sync/atomic"
)

type Console interface {
	io.Closer
	SetLevel(Level)
	Log(Level, ...interface{})
	ReadLine() (string, error)
	WriteError(error)
	WriteLine(string)
}

type console struct {
	dest   *bufio.Writer
	err    *bufio.Writer
	lines  chan *lineData
	level  Level
	mutex  sync.Mutex
	closed int32
}

var _ Console = (*console)(nil)

type lineData struct {
	seg    string
	prefix bool
	err    error
}

func NewConsole(level Level) Console {
	c := &console{
		dest:  bufio.NewWriter(os.Stdout),
		err:   bufio.NewWriter(os.Stderr),
		lines: make(chan *lineData),
		level: level,
	}
	go func() {
		src := bufio.NewReader(os.Stdin)
		for {
			line := &lineData{}
			var seg []byte
			seg, line.prefix, line.err = src.ReadLine()
			if line.err == nil {
				line.seg = string(seg)
			}
			c.lines <- line
		}
	}()
	return c
}

func (c *console) SetLevel(level Level) {
	c.level = level
}

func (c *console) Log(level Level, args ...interface{}) {
	if level < c.level || len(args) < 1 {
		return
	}
	if level > Warning {
		if cast, ok := args[0].(error); ok && len(args) == 1 {
			c.WriteError(cast)
		} else {
			c.WriteError(errors.New(fmt.Sprintln(args...)))
		}
	} else {
		c.WriteLine(fmt.Sprintln(args...))
	}
}

func (c *console) ReadLine() (string, error) {
	out := ""
	for line := range c.lines {
		if line.err != nil {
			return "", line.err
		}
		if !line.prefix {
			return out + line.seg, nil
		}
		out += string(line.seg)
	}
	return out, io.EOF
}

func (c *console) WriteLine(msg string) {
	c.mutex.Lock()
	defer c.mutex.Unlock()
	_, _ = c.dest.WriteString(msg)
	length := len(msg)
	if length < 1 || msg[length-1] != '\n' {
		_ = c.dest.WriteByte('\n')
	}
	c.dest.Flush()
}

func (c *console) WriteError(err error) {
	c.mutex.Lock()
	defer c.mutex.Unlock()
	_, _ = c.err.WriteString(err.Error())
	_ = c.err.WriteByte('\n')
	c.err.Flush()
}

func (c *console) Close() error {
	if !atomic.CompareAndSwapInt32(&c.closed, 0, 1) {
		return nil
	}
	c.mutex.Lock()
	defer c.mutex.Unlock()
	c.dest.Flush()
	c.err.Flush()
	close(c.lines)
	return nil
}
