package datadog

import (
	"fmt"
	"sort"
	"strings"
	"sync"

	"code.justin.tv/devhub/e2ml/libs/metrics"
	datadog "github.com/DataDog/datadog-go/statsd"
)

type tracker struct {
	client      *datadog.Client
	counts      map[string]*count
	timings     map[string]*timing
	gauges      map[string]*gauge
	aggregators map[string]*aggregator
	mutex       sync.RWMutex
}

func NewTracker(host, namespace string, src metrics.TagSource) (metrics.Tracker, error) {
	if !strings.Contains(host, ":") {
		host = host + ":8125"
	}
	client, err := datadog.New(
		host,
		datadog.WithNamespace(namespace),
		datadog.WithTags(src.GetTags()),
		datadog.Buffered(),
		datadog.WithMaxMessagesPerPayload(32),
	)
	if err != nil {
		return nil, err
	}
	return &tracker{
		client:      client,
		counts:      make(map[string]*count),
		timings:     make(map[string]*timing),
		gauges:      make(map[string]*gauge),
		aggregators: make(map[string]*aggregator),
	}, nil
}

func (*tracker) String() string { return "{datadog metrics tracker}" }

func (t *tracker) Tick() {
	t.mutex.RLock()
	defer t.mutex.RUnlock()
	for _, g := range t.gauges {
		g.tick()
	}
	for _, a := range t.aggregators {
		a.tick()
	}
	for _, c := range t.counts {
		c.tick()
	}
}

func (t *tracker) Close() error {
	t.Tick()
	err := t.client.Flush()
	t.client.Close()
	return err
}

func (t *tracker) Aggregator(name string, tags []string) metrics.Aggregator {
	key := createKey(name, tags)
	t.mutex.RLock()
	a, ok := t.aggregators[key]
	t.mutex.RUnlock()
	if ok {
		return a
	}
	t.mutex.Lock()
	a, ok = t.aggregators[key]
	if !ok {
		a = &aggregator{name, tags, t, 0}
		t.aggregators[key] = a
	}
	t.mutex.Unlock()
	return a
}

func (t *tracker) Count(name string, tags []string) metrics.Count {
	key := createKey(name, tags)
	t.mutex.RLock()
	c, ok := t.counts[key]
	t.mutex.RUnlock()
	if ok {
		return c
	}
	t.mutex.Lock()
	c, ok = t.counts[key]
	if !ok {
		c = &count{name, tags, t, 0}
		t.counts[key] = c
	}
	t.mutex.Unlock()
	return c
}

func (t *tracker) Timing(name string, tags []string) metrics.Timing {
	key := createKey(name, tags)
	t.mutex.RLock()
	tm, ok := t.timings[key]
	t.mutex.RUnlock()
	if ok {
		return tm
	}
	t.mutex.Lock()
	tm, ok = t.timings[key]
	if !ok {
		tm = &timing{name, tags, t}
		t.timings[key] = tm
	}
	t.mutex.Unlock()
	return tm
}

func (t *tracker) Gauge(name string, tags []string) metrics.Gauge {
	key := createKey(name, tags)
	t.mutex.RLock()
	g, ok := t.gauges[key]
	t.mutex.RUnlock()
	if ok {
		return g
	}
	t.mutex.Lock()
	g, ok = t.gauges[key]
	if !ok {
		g = &gauge{name, tags, t, 0}
		t.gauges[key] = g
	}
	t.mutex.Unlock()
	return g
}

func createKey(name string, tags []string) string {
	if len(tags) == 0 {
		return name
	}
	sorted := make([]string, len(tags))
	copy(sorted, tags)
	sort.Strings(sorted)
	return fmt.Sprintf("%s:%v", name, strings.Join(sorted, "."))
}
