package metrics

import (
	"io"
	"time"
)

// Tracker allows construction of metric objects that are optimized for frequent
// use; requests for identical parameters will yield a cached object, but the
// returned objects should be stored and referenced directly for efficiency.
type Tracker interface {
	io.Closer
	Tick()
	Aggregator(name string, tags []string) Aggregator
	Count(name string, tags []string) Count
	Gauge(name string, tags []string) Gauge
	Timing(name string, tags []string) Timing
}

// Count is used to measure occurrences over time
type Count interface {
	Add(int64)
}

// Gauge is used to report the current value of a variable
type Gauge interface {
	Set(value float64)
}

// Aggregator converts a count-like interface to a gauge
type Aggregator interface {
	Add(value int64)
	Set(value float64)
}

// Timing is used to report how long something took
type Timing interface {
	Sample(value time.Duration)
}
