package promise

import (
	"sync"
	"sync/atomic"
)

type Any interface {
	WouldBlock() bool
	Get() (interface{}, error)
}

type MutableAny interface {
	Any
	Set(interface{}, error)
}

type any struct {
	data interface{}
	err  error
}

func NewSyncAny(data interface{}, err error) Any { return &any{data, err} }
func (s *any) WouldBlock() bool                  { return false }
func (s *any) Get() (interface{}, error)         { return s.data, s.err }

type async struct {
	hasSet int32
	data   interface{}
	err    error
	loaded sync.WaitGroup
}

func NewAny() MutableAny {
	a := &async{}
	a.loaded.Add(1)
	return a
}

func (a *async) WouldBlock() bool {
	return atomic.LoadInt32(&a.hasSet) == 0
}

func (a *async) Get() (interface{}, error) {
	a.loaded.Wait()
	return a.data, a.err
}

func (a *async) Set(data interface{}, err error) {
	if atomic.SwapInt32(&a.hasSet, 1) == 0 {
		a.data = data
		a.err = err
		a.loaded.Done()
	}
}
