package stream

import (
	"fmt"
	"sort"
	"strings"
)

// Namespace describes a discovery target for sources
type Namespace string

// NoAddress signifies that no address has been chosen
const NoAddress = AddressKey("")

// Version provides a formatting clue to Sources, allowing graceful cutover of
// content formats and detection of out-of-date clients
type Version uint16

// An Address describes a unique name for a message stream; when an address is
// passed to a source it is possible that for efficiency reasons the resulting
// stream may use a parent scope of the indicated path; inbound messages should
// be checked against explicitly subscribed paths before they are used.
//
// Address filters within a namespace provide a multidimentional area-of-interest
// declaration that should be used to restrict outgoing content. Interservice
// communication from subscription sources or within the cluster may be optimzed
// by sending unfiltered content for low frequency data or highly partitioned
// namespaces with a large number of distinct subscriptions to parse. The gateway
// will always honor filtering declared by its clients.
//
// When considering the validity of a message for an address, the filter maps
// must have a nonzero intersection of keys with no conflict in values. For any
// namespace that has a single filter dimension, this is the same as an exact
// match and Key() comparison becomes possible.  Stream sequences and missing
// message tracking are always maintained at message addresses rather than the
// subscription addresses provided by listeners; for this reason it is necessary
// for a client to maintain a map of last seen offset per incoming address.
type Address interface {
	AddressScope                                     // allow address comparison/nesting
	Namespace() Namespace                            // used for discovering a source for stream information
	Version() Version                                // format version number for this particular namespace
	Filter(string) (string, bool)                    // optimized query for a filter field
	WithFilter(field, value string) (Address, error) // overwrite or append a filter in a copy of this address
}

// NewAddress constructs an optimized address given the namespace/version/filters
func NewAddress(nspace Namespace, version Version, filters map[string]string) (Address, error) {
	if strings.ContainsAny(string(nspace), reserved) {
		return nil, ErrReservedCharacters
	}
	if len(nspace) == 0 {
		return nil, ErrMissingRequiredNamespace
	}
	if len(filters) > maxFilters {
		return nil, ErrTooManyFilters
	}
	switch len(filters) {
	case 0:
		return newGlobalAddress(nspace, version, ""), nil
	case 1:
		for k, v := range filters {
			if strings.ContainsAny(k, reserved) || strings.ContainsAny(v, reserved) {
				return nil, ErrReservedCharacters
			}
			return newSingleFilterAddress(nspace, version, k, v, ""), nil
		}
	}
	segments := make([]string, 0, len(filters))
	copy := make(map[string]string)
	for k, v := range filters {
		if strings.ContainsAny(k, reserved) || strings.ContainsAny(v, reserved) {
			return nil, ErrReservedCharacters
		}
		segments = append(segments, k+"="+v)
		copy[k] = v
	}
	sort.Strings(segments)
	key := fmt.Sprintf("%s@%d?%s", nspace, version, strings.Join(segments, "&"))
	return &multiFilterAddress{
		nspace:  nspace,
		version: version,
		filters: copy,
		key:     AddressKey(key),
	}, nil
}

type globalAddress struct {
	nspace  Namespace
	version Version
	key     AddressKey
}

func newGlobalAddress(nspace Namespace, ver Version, key AddressKey) Address {
	if key == "" {
		key = AddressKey(fmt.Sprintf("%s@%d", nspace, ver))
	}
	return &globalAddress{nspace, ver, key}
}

func (g *globalAddress) Namespace() Namespace               { return g.nspace }
func (g *globalAddress) Version() Version                   { return g.version }
func (g *globalAddress) Filter(field string) (string, bool) { return "", false }
func (g *globalAddress) Cardinality() int                   { return 1 }
func (g *globalAddress) Key() AddressKey                    { return g.key }
func (m *globalAddress) String() string                     { return "^" + string(m.key) }
func (g *globalAddress) Parents() AddressScopes             { return anyAddressSlice }
func (g *globalAddress) Includes(addr Address) bool {
	return addr.Version() == g.version && addr.Namespace() == g.nspace
}
func (g *globalAddress) WithFilter(field, value string) (Address, error) {
	return NewAddress(g.nspace, g.version, map[string]string{field: value})
}

type singleFilterAddress struct {
	nspace  Namespace
	version Version
	field   string
	value   string
	key     AddressKey
}

func newSingleFilterAddress(nspace Namespace, ver Version, field, value string, key AddressKey) Address {
	if key == "" {
		key = AddressKey(fmt.Sprintf("%s@%d?%s=%s", nspace, ver, field, value))
	}
	return &singleFilterAddress{nspace, ver, field, value, key}
}
func (s *singleFilterAddress) Namespace() Namespace               { return s.nspace }
func (s *singleFilterAddress) Version() Version                   { return s.version }
func (s *singleFilterAddress) Filter(field string) (string, bool) { return s.value, field == s.field }
func (s *singleFilterAddress) Cardinality() int                   { return 2 }
func (s *singleFilterAddress) Key() AddressKey                    { return s.key }
func (s *singleFilterAddress) String() string                     { return "^" + string(s.key) }
func (s *singleFilterAddress) Parents() AddressScopes {
	return AddressScopes{newGlobalAddress(s.nspace, s.version, "")}
}
func (s *singleFilterAddress) Includes(addr Address) bool {
	if addr.Key() == s.key {
		return true
	}
	value, ok := addr.Filter(s.field)
	return ok && value == s.value && addr.Version() == s.version && addr.Namespace() == s.nspace
}
func (s *singleFilterAddress) WithFilter(field, value string) (Address, error) {
	return NewAddress(s.nspace, s.version, map[string]string{s.field: s.value, field: value})
}

type multiFilterAddress struct {
	nspace  Namespace
	version Version
	filters map[string]string
	key     AddressKey
}

// NOTE : we always rebuild the key here to ensure it's canonicalized
func newMultiFilterAddress(nspace Namespace, ver Version, filters map[string]string) Address {
	segments := make([]string, 0, len(filters))
	copy := make(map[string]string)
	for k, v := range filters {
		segments = append(segments, k+"="+v)
		copy[k] = v
	}
	sort.Strings(segments) // sort into canonical
	key := fmt.Sprintf("%s@%d?%s", nspace, ver, strings.Join(segments, "&"))
	return &multiFilterAddress{Namespace(nspace), Version(ver), filters, AddressKey(key)}
}

func (m *multiFilterAddress) Namespace() Namespace { return m.nspace }
func (m *multiFilterAddress) Version() Version     { return m.version }
func (m *multiFilterAddress) Cardinality() int     { return len(m.filters) + 1 }
func (m *multiFilterAddress) Key() AddressKey      { return m.key }
func (m *multiFilterAddress) String() string       { return "^" + string(m.key) }
func (m *multiFilterAddress) Filter(field string) (string, bool) {
	v, ok := m.filters[field]
	return v, ok
}

func (m *multiFilterAddress) WithFilter(field, value string) (Address, error) {
	filters := make(map[string]string)
	for f, v := range m.filters {
		filters[f] = v
	}
	filters[field] = value
	return NewAddress(m.nspace, m.version, filters)
}

// Parents should return the list of addresses that include this one with a
// decremented cardinality
func (m *multiFilterAddress) Parents() AddressScopes {
	scopes := []AddressScope{}
	switch len(m.filters) {
	case 2:
		for k, v := range m.filters {
			scopes = append(scopes, newSingleFilterAddress(m.nspace, m.version, k, v, ""))
		}
	default:
		// build list of parent filters by skipping one field each; store by missing field
		parentFilters := make(map[string]map[string]string)
		for k := range m.filters {
			parentFilters[k] = make(map[string]string)
		}
		for k, v := range m.filters {
			for o, t := range parentFilters {
				if o == k {
					continue
				}
				t[k] = v
			}
		}
		for _, filter := range parentFilters {
			scopes = append(scopes, newMultiFilterAddress(m.nspace, m.version, filter))
		}
	}
	return scopes
}

func (m *multiFilterAddress) Includes(addr Address) bool {
	if addr.Key() == m.key {
		return true
	}
	if m.version != addr.Version() || m.nspace != addr.Namespace() {
		return false
	}
	for k, v := range m.filters {
		if value, ok := addr.Filter(k); !ok || value != v {
			return false
		}
	}
	return true
}
