package stream

import (
	"bytes"
	"sort"
	"strings"
)

const (
	anyAddressKey  = AddressKey("*")
	NoAddressMatch = -1
)

var scopeDelimiter = []byte{'\000'}

// An AddressScope defines a boundary for related addresses; it is used to
// resolve inter-address filtering relationships and answer the requestion of
// whether a particular service can serve an address.
//
// The Key() function is used for equality checking addresses as an exact match
// which allows efficient grouping of memberships.
type AddressScope interface {
	Cardinality() int       // how exact is this scope? Higher numbers match fewer addresses
	Includes(Address) bool  // true if the address matches this scope
	Key() AddressKey        // used to determine if addresses are an exact match
	Parents() AddressScopes // list of matching scopes with Cardinality N-1 (empty at Cardinality 0)
	String() string         // provide debug friendly %v
}

// AnyAddress is a special case of address scope that matches any legal address
var AnyAddress = anyAddress{}
var anyAddressSlice = AddressScopes{AnyAddress}

type anyAddress struct{}

func (anyAddress) Cardinality() int       { return 0 }
func (anyAddress) Includes(Address) bool  { return true }
func (anyAddress) Key() AddressKey        { return anyAddressKey }
func (anyAddress) String() string         { return "^" + string(anyAddressKey) }
func (anyAddress) Parents() AddressScopes { return nil }

// ParseAddressScope attempts to construct an address scope from a string
func ParseAddressScope(key string) (AddressScope, error) {
	if AddressKey(key) == anyAddressKey {
		return AnyAddress, nil
	}
	return ParseAddress(key)
}

type AddressScopes []AddressScope

var _ sort.Interface = AddressScopes(nil)

func (a AddressScopes) Contains(scope AddressScope) bool {
	key := scope.Key()
	for _, elem := range a {
		if elem.Key() == key {
			return true
		}
	}
	return false
}

// AddressScopes are sortable
func (a AddressScopes) Len() int { return len(a) }
func (a AddressScopes) Less(i, j int) bool {
	comp := a[j].Cardinality() - a[i].Cardinality() // descending order
	if comp == 0 {
		comp = strings.Compare(string(a[i].Key()), string(a[j].Key()))
	}
	return comp < 0
}

func (a AddressScopes) Swap(i, j int) {
	a[i], a[j] = a[j], a[i]
}

// AddressScopes can be binary marshalled
func (a AddressScopes) MarshalBinary() ([]byte, error) {
	data := make([][]byte, len(a))
	for i, scope := range a {
		data[i] = []byte(scope.Key())
	}
	return bytes.Join(data, scopeDelimiter), nil
}

func (a *AddressScopes) UnmarshalBinary(data []byte) error {
	if len(data) == 0 {
		*a = AddressScopes{}
		return nil
	}

	keys := bytes.Split(data, scopeDelimiter)
	out := make(AddressScopes, len(keys))
	for i, key := range keys {
		addr, err := ParseAddressScope(string(key))
		if err != nil {
			return err
		}
		out[i] = addr
	}
	*a = out
	return nil
}

// HasBetterMatch returns true if the scopes match an address with better
// cardinality than the minimum specified value. Must be called on a descending
// sorted list.
func (a AddressScopes) HasBetterMatch(address Address, minCardinality int) (int, bool) {
	for _, scope := range a {
		cardinality := scope.Cardinality()
		if cardinality <= minCardinality {
			break
		}
		if scope.Includes(address) {
			return cardinality, true
		}
	}
	return minCardinality, false
}

// AncestorList takes any scope and returns a sorted list of ancestors
func AncestorList(a AddressScope) AddressScopes {
	set := make(map[AddressKey]struct{})
	result := AddressScopes{a}
	for index := 0; index < len(result); index += 1 {
		for _, scope := range result[index].Parents() {
			key := scope.Key()
			if _, found := set[key]; !found {
				result = append(result, scope)
				set[key] = struct{}{}
			}
		}
	}
	result = result[1:]
	sort.Sort(result)
	return result
}
