package s2s

import (
	"fmt"
	"time"

	"code.justin.tv/common/jwt"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

type OpaqueBytes = stream.OpaqueBytes

// NewAuthSource returns a fatory to generate s2s requests
func NewAuthSource(secret OpaqueBytes, iss, aud string) (stream.AuthSource, error) {
	if err := ValidateSecretLen(secret); err != nil {
		return nil, err
	}
	algorithm := jwt.HS512(secret)
	header := jwt.NewHeader(algorithm)

	return func() stream.AuthRequest {
		now := time.Now()
		jwtClaims := NewJWTClaims(now, iss, aud)
		encodedJWT, err := jwt.Encode(header, jwtClaims, algorithm)
		if err != nil {
			// encoding is done from known values, errors should not happen
			encodedJWT = []byte{}
		}

		return &request{encodedJWT: OpaqueBytes(encodedJWT)}
	}, nil
}

func ValidateSecretLen(secret OpaqueBytes) error {
	const SECRET_MIN_BYTES = 64 // 512 bits
	if len(secret) < SECRET_MIN_BYTES {
		return fmt.Errorf("s2s-secret is empty or has less than %d bytes", SECRET_MIN_BYTES)
	}
	return nil
}
