package s2s

import (
	"net/http"
	"strings"

	"code.justin.tv/common/jwt"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/auth"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

type extractor struct{}

var extSingleton = &extractor{}

func NewExtractor() auth.Extractor { return extSingleton }

func (e *extractor) Method() string { return s2sMethod }

func (e *extractor) Extract(h http.Header) (stream.AuthRequest, bool) {
	encodedJWT, jwtParts, err := httpHeadersExtractAndParse(h)
	if err != nil {
		return nil, false
	}
	return &request{
		encodedJWT: encodedJWT,
		parsedJWT:  jwtParts,
	}, true
}

func (e *extractor) FailureReason(h http.Header) error {
	_, _, err := httpHeadersExtractAndParse(h)
	return err
}

func httpHeadersExtractAndParse(h http.Header) ([]byte, *jwt.Fragment, error) {
	val := h.Get(protocol.HTTPAuthorizationHeader)
	if val == "" {
		return nil, nil, stream.ErrMissingAuthorization
	}
	if !strings.HasPrefix(val, protocol.HTTPBearerPrefix) {
		return nil, nil, stream.ErrInvalidAuthHeaders
	}

	encodedJWTStr := val[len(protocol.HTTPBearerPrefix):]
	encodedJWT := []byte(encodedJWTStr)
	jwtParts, err := jwt.Parse(encodedJWT)
	if err != nil {
		return nil, nil, stream.ErrInvalidJWT
	}

	return encodedJWT, &jwtParts, nil
}

func httpHeadersInject(h http.Header, jwtToken string) {
	h.Set(protocol.HTTPAuthorizationHeader, protocol.HTTPBearerPrefix+jwtToken)
}
