package s2s

import (
	"time"

	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	s2sMethod         = "s2s"
	jwtExpirationTime = 1 * time.Minute
)

func NewJWTClaims(now time.Time, iss, aud string) jwtClaims {
	return jwtClaims{
		Iss: iss,
		Aud: aud,
		Iat: now.Unix(),
		Exp: now.Add(jwtExpirationTime).Unix(),
	}
}

type jwtClaims struct {
	Iss string `json:"iss"` // principal that issued the JWT (caller/client)
	Aud string `json:"aud"` // audience (receiver/host)
	Iat int64  `json:"iat"` // issued at
	Exp int64  `json:"exp"` // expiration
}

func (jwtc jwtClaims) IatAsTime() time.Time { return time.Unix(jwtc.Iat, 0) }
func (jwtc jwtClaims) ExpAsTime() time.Time { return time.Unix(jwtc.Exp, 0) }

func (jwtc jwtClaims) ValidateFields(now time.Time, expectedAud string) error {
	if now.After(jwtc.ExpAsTime()) {
		return stream.ErrAuthExpired
	}
	if jwtc.Aud != expectedAud {
		return stream.ErrAuthAudienceMismatch
	}
	return nil
}
