package s2s

import (
	"time"

	"code.justin.tv/common/jwt"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/auth"
)

// NewResolver resolves and validates credentials from a decoded s2s auth request.
func NewResolver(secret OpaqueBytes, aud string) (auth.Resolver, error) {
	if err := ValidateSecretLen(secret); err != nil {
		return nil, err
	}
	return &Resolver{
		secret:      secret,
		algorithm:   jwt.HS512(secret),
		expectedAud: aud,
	}, nil
}

type Resolver struct {
	secret      []byte // shared secret that was used to encode and decode the JWT token
	algorithm   jwt.Algorithm
	expectedAud string // audience is the current service name, used to validate the JWT claim set by the caller
}

func (r *Resolver) Method() string { return s2sMethod }

func DecodeAndValidateJWT(secret, encodedJWT []byte) (jwtClaims, error) {
	algorithm := jwt.HS512(secret)
	var header jwt.Header
	var jwtc jwtClaims
	err := jwt.DecodeAndValidate(&header, &jwtc, algorithm, encodedJWT)
	return jwtc, err
}

func (r *Resolver) Resolve(authReq stream.AuthRequest) (stream.Credentials, error) {
	req, ok := authReq.(*request)
	if !ok {
		return nil, stream.ErrUnsupportedAuthMethod
	}

	// Parse if needed
	jwtParts := req.parsedJWT // already parsed by Decoder/Extractor
	if jwtParts == nil {      // otherwise parse from the encoded token
		f, err := jwt.Parse(req.encodedJWT)
		if err != nil {
			return nil, stream.ErrInvalidJWT
		}
		jwtParts = &f
	}

	// Validate signature with algorithm
	if err := jwtParts.Validate(r.algorithm); err != nil {
		return nil, stream.ErrInvalidSignature
	}

	// Decode
	var header jwt.Header
	var jwtc jwtClaims
	if err := jwtParts.Decode(&header, &jwtc); err != nil {
		return nil, stream.ErrInvalidJWTClaims
	}

	// Ensure decoded claims have expected values
	now := time.Now()
	if err := jwtc.ValidateFields(now, r.expectedAud); err != nil {
		return nil, err
	}

	// Return valid credentials. No need to have special permissions for different services yet,
	// the individual client/message authotization will take care of that.
	return stream.AllPermissions(), nil
}
