package stream

import (
	"bytes"
	"encoding"
	"encoding/binary"
	"fmt"
	"sort"
	"time"
)

var (
	credentialMagic        = []byte("Cr3\001")
	credentialDelimiter    = []byte("\000\000")
	credentialDelimiterLen = len(credentialDelimiter)
)

type Credentials interface {
	encoding.BinaryMarshaler
	ClientID() string
	CanListen(Address) bool
	CanSend(Address) bool
	Expires() *time.Time
	String() string
}

type credentials struct {
	clientID string
	listen   AddressScopes
	send     AddressScopes
	expires  *time.Time
}

var (
	allScopes = AddressScopes{AnyAddress}
	all       = NewCredentials("*", allScopes, allScopes)
	none      = NewCredentials("-", AddressScopes{}, AddressScopes{})
)

func AllPermissions() Credentials { return all }
func NoPermissions() Credentials  { return none }

func NewTimedCredentials(clientID string, listen, send AddressScopes, expires time.Time) Credentials {
	return newCredentials(clientID, listen, send, &expires)
}

func NewCredentials(clientID string, listen, send AddressScopes) Credentials {
	return newCredentials(clientID, listen, send, nil)
}

func newCredentials(clientID string, listen, send AddressScopes, expires *time.Time) Credentials {
	l := make(AddressScopes, len(listen))
	copy(l, listen)
	sort.Sort(l)
	s := make(AddressScopes, len(send))
	copy(s, send)
	sort.Sort(s)
	return &credentials{clientID, l, s, expires}
}

func (c *credentials) Expires() *time.Time { return c.expires }
func (c *credentials) ClientID() string    { return c.clientID }
func (c *credentials) CanListen(addr Address) bool {
	if c.expires != nil && c.expires.Before(time.Now()) {
		return false
	}
	_, ok := c.listen.HasBetterMatch(addr, NoAddressMatch)
	return ok
}

func (c *credentials) CanSend(addr Address) bool {
	if c.expires != nil && c.expires.Before(time.Now()) {
		return false
	}
	_, ok := c.send.HasBetterMatch(addr, NoAddressMatch)
	return ok
}

func (c *credentials) MarshalBinary() ([]byte, error) {
	lbytes, _ := c.listen.MarshalBinary()
	sbytes, _ := c.send.MarshalBinary()
	ebytes := []byte{}
	if c.expires != nil {
		ebytes = make([]byte, 8)
		binary.BigEndian.PutUint64(ebytes, uint64(c.expires.UnixNano()))
	}
	return bytes.Join([][]byte{credentialMagic, []byte(c.clientID), lbytes, sbytes, ebytes}, credentialDelimiter), nil
}

func UnmarshalCredentials(data []byte) (Credentials, error) {
	end := bytes.Index(data, credentialDelimiter)
	if end < 0 || !bytes.Equal(data[:end], credentialMagic) {
		return nil, ErrInvalidCredentialFormat
	}

	var listen, send AddressScopes

	start, end, ok := nextSegment(data, end)
	if !ok {
		return nil, ErrInvalidCredentialFormat
	}
	clientID := string(data[start:end])

	if start, end, ok = nextSegment(data, end); !ok {
		return nil, ErrInvalidCredentialFormat
	}

	if err := listen.UnmarshalBinary(data[start:end]); err != nil {
		return nil, err
	}

	if start, end, ok = nextSegment(data, end); !ok {
		return nil, ErrInvalidCredentialFormat
	}

	if err := send.UnmarshalBinary(data[start:end]); err != nil {
		return nil, err
	}

	var expires *time.Time
	start = end + credentialDelimiterLen
	switch len(data[start:]) {
	case 0:
	case 8:
		time := time.Unix(0, int64(binary.BigEndian.Uint64(data[start:])))
		expires = &time
	default:
		return nil, ErrInvalidCredentialFormat
	}

	return newCredentials(clientID, listen, send, expires), nil
}

func (c *credentials) String() string {
	return fmt.Sprintf("{id: %v listen: %v send: %v expires: %v}", c.clientID, c.listen, c.send, c.expires)
}

func nextSegment(raw []byte, prevEnd int) (int, int, bool) {
	start := prevEnd + credentialDelimiterLen
	if start >= len(raw) {
		return 0, 0, false
	}
	end := bytes.Index(raw[start:], credentialDelimiter) + start
	if end < start {
		return 0, 0, false
	}
	return start, end, true
}
