package stream

import (
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/devhub/e2ml/libs/errors"
)

const (
	reserved   = "@?&=\000"
	maxFilters = 4
)

const (
	ErrForbiddenCode                = "access_forbidden"
	ErrInvalidAuthorizationCode     = "invalid_authorization"
	ErrInvalidCredentialFormatCode  = "invalid_credential_format"
	ErrInvalidNamespaceCode         = "invalid_address_namespace"
	ErrMissingAuthorizationCode     = "missing_authorization"
	ErrMissingFilterValueCode       = "missing_filter_value"
	ErrMissingRequiredNamespaceCode = "missing_namespace"
	ErrMissingRequiredVersionCode   = "missing_version"
	ErrNoProgressCode               = "no_progress"
	ErrNotImplementedCode           = "not_implemented"
	ErrReservedCharactersCode       = "invalid_characters"
	ErrTooManyFiltersCode           = "too_many_address_filters"
	ErrVersionOutOfRangeCode        = "invalid_version_number"
	ErrVersionSyntaxCode            = "invalid_version_syntax"
	ErrReaderAlreadyClosedCode      = "reader_already_closed"
	ErrWriterAlreadyClosedCode      = "writer_already_closed"

	AuthErrCodeAudienceMismatch         = "audience_mismatch"
	AuthErrCodeBadMethod                = "bad_method"
	AuthErrCodeExpectedAnonymous        = "expected_anonymous"
	AuthErrCodeExpired                  = "expired"
	AuthErrCodeInvalidChannelID         = "invalid_channel_id"
	AuthErrCodeInvalidClientID          = "invalid_client_id"
	AuthErrCodeInvalidJWTClaims         = "invalid_jwt_claims"
	AuthErrCodeInvalidSignature         = "invalid_signature"
	AuthErrCodeMalformedHeader          = "malformed_header"
	AuthErrCodeMalformedJWT             = "malformed_jwt"
	AuthErrCodeMissingChannelID         = "missing_channel_id"
	AuthErrCodeMissingClientID          = "missing_client_id"
	AuthErrCodeMissingJWTClaims         = "missing_jwt_claims"
	AuthErrCodeMissingListenPermissions = "missing_listen_permissions"
	AuthErrCodeUnupportedMethod         = "unsupported_method"
)

type commonSupport interface {
	error
	errors.ErrorCodeSource
	errors.HTTPStatusSource
	errors.DetailsSource
}
type missingFilterValue struct{ filter string }

var _ commonSupport = (*missingFilterValue)(nil)

func (m *missingFilterValue) Error() string {
	return fmt.Sprintf("Address filter %s must have a value", m.filter)
}
func (m *missingFilterValue) ErrorCode() string       { return ErrMissingFilterValueCode }
func (m *missingFilterValue) Details() errors.Details { return errors.Details{"filter": m.filter} }
func (m *missingFilterValue) HTTPStatus() int         { return http.StatusUnprocessableEntity }
func ErrMissingFilterValue(filter string) error       { return &missingFilterValue{filter} }
func ErrMissingFilterValueBuilder(d errors.Details) error {
	return ErrMissingFilterValue(d.String("filter"))
}

type InvalidAuthorizationError interface {
	error
	Reason() string
}

type invalidAuthorization struct {
	reason string
}

var reasonStrings = map[string]string{
	AuthErrCodeAudienceMismatch:         "The JWT audience does not match",
	AuthErrCodeBadMethod:                "The auth method was not valid",
	AuthErrCodeExpectedAnonymous:        "This endpoint expected an anonymous request",
	AuthErrCodeExpired:                  "The authorization has expired",
	AuthErrCodeInvalidChannelID:         "The channel ID was not valid",
	AuthErrCodeInvalidClientID:          "The client ID was not valid",
	AuthErrCodeInvalidJWTClaims:         "The JWT claims have unexpected JSON types",
	AuthErrCodeInvalidSignature:         "The signature was invalid",
	AuthErrCodeMalformedHeader:          "The auth header syntax is incorrect",
	AuthErrCodeMalformedJWT:             "The JWT syntax is incorrect",
	AuthErrCodeMissingChannelID:         "The channel ID was missing from the JWT",
	AuthErrCodeMissingClientID:          "The client ID was missing from headers",
	AuthErrCodeMissingJWTClaims:         "The JWT has missing claims",
	AuthErrCodeMissingListenPermissions: "At least one listen pemission is required",
	AuthErrCodeUnupportedMethod:         "This auth method is not supported",
}

var _ commonSupport = (*invalidAuthorization)(nil)

func (*invalidAuthorization) resolveReason(reason string) string {
	if str, ok := reasonStrings[reason]; ok {
		return str
	}
	return reason
}

func (i *invalidAuthorization) Error() string {
	return "Invalid Authorization: " + i.resolveReason(i.reason)
}
func (i *invalidAuthorization) ErrorCode() string { return ErrInvalidAuthorizationCode }
func (i *invalidAuthorization) Reason() string    { return i.reason }
func (i *invalidAuthorization) Details() errors.Details {
	return errors.Details{"reason": i.reason}
}
func (i *invalidAuthorization) HTTPStatus() int { return http.StatusForbidden }
func ErrInvalidAuthorization(reason string) InvalidAuthorizationError {
	return &invalidAuthorization{reason}
}
func ErrInvalidAuthorizationBuilder(d errors.Details) error {
	return ErrInvalidAuthorization(d.String("reason"))
}

var internalList = []errors.ErrorCodeError{}

func quickBuild(code string, status int, msg string) error {
	err := errors.NewBuilder(msg).WithHTTPStatus(status).Build().WithErrorCode(code)
	internalList = append(internalList, err)
	return err
}

var (
	ErrForbidden                = quickBuild(ErrForbiddenCode, http.StatusForbidden, "Access forbidden")
	ErrInvalidCredentialFormat  = quickBuild(ErrInvalidCredentialFormatCode, http.StatusUnprocessableEntity, "The credentials were not formatted correctly")
	ErrInvalidNamespace         = quickBuild(ErrInvalidNamespaceCode, http.StatusUnprocessableEntity, "The requested address namespace was not valid")
	ErrMissingAuthorization     = quickBuild(ErrMissingAuthorizationCode, http.StatusUnauthorized, "The authorization header must be set")
	ErrMissingRequiredNamespace = quickBuild(ErrMissingRequiredNamespaceCode, http.StatusUnprocessableEntity, "The address key must include a namespace")
	ErrMissingRequiredVersion   = quickBuild(ErrMissingRequiredVersionCode, http.StatusUnprocessableEntity, "The address key must include a version")
	ErrNotImplemented           = quickBuild(ErrNotImplementedCode, http.StatusNotImplemented, "This feature not implemented")
	ErrNoProgress               = quickBuild(ErrNoProgressCode, http.StatusInternalServerError, "This listener did not report progress when updated")
	ErrReservedCharacters       = quickBuild(ErrReservedCharactersCode, http.StatusUnprocessableEntity, "The characters "+strconv.Quote(reserved)+" are reserved")
	ErrTooManyFilters           = quickBuild(ErrTooManyFiltersCode, http.StatusUnprocessableEntity, fmt.Sprintf("An address may contain a maximum of %v filters", maxFilters))
	ErrVersionOutOfRange        = quickBuild(ErrVersionOutOfRangeCode, http.StatusUnprocessableEntity, "Address version must be in the range 0-65535")
	ErrVersionSyntax            = quickBuild(ErrVersionSyntaxCode, http.StatusUnprocessableEntity, "Address version must be an integer")
	ErrWriterAlreadyClosed      = quickBuild(ErrWriterAlreadyClosedCode, http.StatusConflict, "Writer has already been closed")
	ErrReaderAlreadyClosed      = quickBuild(ErrReaderAlreadyClosedCode, http.StatusConflict, "Reader has already been closed")

	ErrAuthAudienceMismatch     = ErrInvalidAuthorization(AuthErrCodeAudienceMismatch)
	ErrAuthExpired              = ErrInvalidAuthorization(AuthErrCodeExpired)
	ErrExpectedAnonymousRequest = ErrInvalidAuthorization(AuthErrCodeExpectedAnonymous)
	ErrInvalidAuthHeaders       = ErrInvalidAuthorization(AuthErrCodeMalformedHeader)
	ErrInvalidAuthMethod        = ErrInvalidAuthorization(AuthErrCodeBadMethod)
	ErrInvalidChannelID         = ErrInvalidAuthorization(AuthErrCodeInvalidChannelID)
	ErrInvalidClientID          = ErrInvalidAuthorization(AuthErrCodeInvalidClientID)
	ErrInvalidJWT               = ErrInvalidAuthorization(AuthErrCodeMalformedJWT)
	ErrInvalidJWTClaims         = ErrInvalidAuthorization(AuthErrCodeInvalidJWTClaims)
	ErrInvalidSignature         = ErrInvalidAuthorization(AuthErrCodeInvalidSignature)
	ErrMissingChannelID         = ErrInvalidAuthorization(AuthErrCodeMissingChannelID)
	ErrMissingClientID          = ErrInvalidAuthorization(AuthErrCodeMissingClientID)
	ErrMissingListenPermissions = ErrInvalidAuthorization(AuthErrCodeMissingListenPermissions)
	ErrMissingJWTClaims         = ErrInvalidAuthorization(AuthErrCodeMissingJWTClaims)
	ErrUnsupportedAuthMethod    = ErrInvalidAuthorization(AuthErrCodeUnupportedMethod)

	Errors = func() errors.Dictionary {
		var builder = errors.NewDictionaryBuilder()
		builder.Include(internalList...)
		builder.Map(ErrMissingFilterValueCode, ErrMissingFilterValueBuilder)
		builder.Map(ErrInvalidAuthorizationCode, ErrInvalidAuthorizationBuilder)
		return builder.Build()
	}()
)
