package stream

import (
	"encoding"
	"encoding/json"
	"strconv"
)

// OpaqueBytes is a safety wrapper around a sensitive value
// that prevents it from being accidentally logged. Unwrap
// the value to transmit it, but only in situations where
// the transmission is known to be secure
type OpaqueBytes []byte

const protectedReplacement = "[secret]"

var _ encoding.TextUnmarshaler = (*OpaqueBytes)(nil)
var _ encoding.TextMarshaler = OpaqueBytes(nil)
var _ encoding.BinaryUnmarshaler = (*OpaqueBytes)(nil)
var _ encoding.BinaryMarshaler = OpaqueBytes(nil)
var _ json.Unmarshaler = (*OpaqueBytes)(nil)
var _ json.Marshaler = OpaqueBytes(nil)

func (o OpaqueBytes) IsEmpty() bool { return len(o) == 0 }

func (o *OpaqueBytes) UnmarshalBinary(value []byte) error {
	*o = make([]byte, len(value))
	copy(*o, value)
	return nil
}

func (o *OpaqueBytes) UnmarshalText(value []byte) error {
	*o = make([]byte, len(value))
	copy(*o, value)
	return nil
}

func (o *OpaqueBytes) UnmarshalJSON(value []byte) error {
	stripped, err := strconv.Unquote(string(value))
	if err != nil {
		return err
	}
	*o = make([]byte, len(stripped))
	copy(*o, stripped)
	return nil
}

func (o OpaqueBytes) MarshalBinary() ([]byte, error) { return []byte(o.String()), nil }
func (o OpaqueBytes) MarshalJSON() ([]byte, error)   { return []byte(strconv.Quote(o.String())), nil }
func (o OpaqueBytes) MarshalText() ([]byte, error)   { return []byte(o.String()), nil }
func (o OpaqueBytes) String() string {
	if len(o) == 0 {
		return ""
	}
	return protectedReplacement
}
