package protocol

import (
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/devhub/e2ml/libs/errors"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	ErrAddressMovedCode       = "address_moved"
	ErrDrainingCode           = "service_draining"
	ErrExpiredAccessCode      = "access_expired"
	ErrForbiddenAccessCode    = "access_forbidden"
	ErrInvalidRequestIDCode   = "invalid_request_id"
	ErrInvalidAddressCode     = "invalid_address"
	ErrInvalidAuthMethodCode  = "invalid_auth_method"
	ErrInvalidLengthCode      = "invalid_length"
	ErrInvalidPayloadCode     = "invalid_payload"
	ErrInvalidSegmentCode     = "invalid_stream_segment"
	ErrInvalidVersionCode     = "invalid_protocol_version"
	ErrInvalidHeaderCode      = "invalid_protocol_header"
	ErrInvalidOpCodeCode      = "unknown_message_type"
	ErrInvalidTicketCode      = "invalid_access_ticket"
	ErrEOFCode                = "unexpected_eof"
	ErrRateLimitedCode        = "address_rate_limited"
	ErrServiceUnavailableCode = "service_unavailable"
	ErrServiceTimeoutCode     = "request_timed_out"
)

type commonErrorSupport interface {
	error
	errors.ErrorCodeSource
	errors.DetailsSource
	errors.HTTPStatusSource
}

type errInvalidLength struct {
	expected int
	actual   int
}

var _ commonErrorSupport = (*errInvalidLength)(nil)

func ErrInvalidLength(expected, actual int) error {
	return &errInvalidLength{expected, actual}
}
func ErrInvalidLengthBuilder(d errors.Details) error {
	return &errInvalidLength{int(d.Int64("expected")), int(d.Int64("actual"))}
}
func (e *errInvalidLength) HTTPStatus() int   { return http.StatusBadRequest }
func (e *errInvalidLength) ErrorCode() string { return ErrInvalidLengthCode }
func (e *errInvalidLength) Details() errors.Details {
	return errors.Details{"expected": e.expected, "actual": e.actual}
}
func (e *errInvalidLength) Error() string {
	return fmt.Sprintf("Invalid length (%d < %d)", e.actual, e.expected)
}

type errInvalidPayload struct {
	code OpCode
}

var _ commonErrorSupport = (*errInvalidPayload)(nil)

func ErrInvalidPayload(code OpCode) error {
	return &errInvalidPayload{code}
}
func ErrInvalidPayloadBuilder(d errors.Details) error {
	return ErrInvalidPayload(OpCode(d.Int64("code")))
}
func (e *errInvalidPayload) HTTPStatus() int         { return http.StatusBadRequest }
func (e *errInvalidPayload) ErrorCode() string       { return ErrInvalidPayloadCode }
func (e *errInvalidPayload) Details() errors.Details { return errors.Details{"code": int(e.code)} }
func (e *errInvalidPayload) Error() string {
	return fmt.Sprintf("Invalid payload for OpCode '%v'", e.code)
}

type errRateLimited struct {
	count  int
	period time.Duration
}

var _ commonErrorSupport = (*errRateLimited)(nil)

func ErrRateLimited(count int, period time.Duration) error {
	return &errRateLimited{count, period}
}
func ErrRateLimitedBuilder(d errors.Details) error {
	period, _ := time.ParseDuration(d.String("period"))
	return ErrRateLimited(int(d.Int64("count")), period)
}

func (e *errRateLimited) HTTPStatus() int   { return http.StatusBadRequest }
func (e *errRateLimited) ErrorCode() string { return ErrRateLimitedCode }
func (e *errRateLimited) Details() errors.Details {
	return errors.Details{"count": e.count, "period": e.period.String()}
}
func (e *errRateLimited) Error() string {
	return fmt.Sprintf("Rate limit of %d writes over %v exceeded", e.count, e.period)
}

var internalList = []errors.ErrorCodeError{}

func quickBuild(code string, status int, msg string) errors.ErrorCodeError {
	err := errors.NewBuilder(msg).WithHTTPStatus(status).Build().WithErrorCode(code)
	internalList = append(internalList, err)
	return err
}

var (
	ErrAddressMoved       = quickBuild(ErrAddressMovedCode, http.StatusMovedPermanently, "Service is no longer serving this address")
	ErrDraining           = quickBuild(ErrDrainingCode, http.StatusServiceUnavailable, "Service is draining; no new connections are permitted")
	ErrExpiredAccess      = quickBuild(ErrExpiredAccessCode, http.StatusForbidden, "Access expired")
	ErrForbiddenAddress   = quickBuild(ErrForbiddenAccessCode, http.StatusForbidden, "Access forbidden")
	ErrInvalidRequestID   = quickBuild(ErrInvalidRequestIDCode, http.StatusUnprocessableEntity, "Invalid request id")
	ErrInvalidAddress     = quickBuild(ErrInvalidAddressCode, http.StatusUnprocessableEntity, "Invalid address")
	ErrInvalidAuthMethod  = quickBuild(ErrInvalidAuthMethodCode, http.StatusUnprocessableEntity, "Invalid auth method")
	ErrInvalidSegment     = quickBuild(ErrInvalidSegmentCode, http.StatusUnprocessableEntity, "Invalid stream segment")
	ErrInvalidVersion     = quickBuild(ErrInvalidVersionCode, http.StatusUnprocessableEntity, "Invalid protocol version")
	ErrInvalidHeader      = quickBuild(ErrInvalidHeaderCode, http.StatusBadRequest, "Invalid protocol header")
	ErrInvalidOpCode      = quickBuild(ErrInvalidOpCodeCode, http.StatusUnprocessableEntity, "Unknown message type")
	ErrInvalidTicket      = quickBuild(ErrInvalidTicketCode, http.StatusForbidden, "Invalid access ticket")
	ErrEOF                = quickBuild(ErrEOFCode, http.StatusBadRequest, "Unexpected EOF")
	ErrServiceUnavailable = quickBuild(ErrServiceUnavailableCode, http.StatusServiceUnavailable, "Service unavailable")
	ErrServiceTimeout     = quickBuild(ErrServiceTimeoutCode, http.StatusServiceUnavailable, "Request timed out")

	Errors = func() errors.Dictionary {
		var builder = errors.NewDictionaryBuilder()
		builder.IncludeAll(stream.Errors)
		builder.Include(internalList...)
		builder.Map(ErrInvalidLengthCode, ErrInvalidLengthBuilder)
		builder.Map(ErrInvalidPayloadCode, ErrInvalidPayloadBuilder)
		builder.Map(ErrRateLimitedCode, ErrRateLimitedBuilder)
		return builder.Build()
	}()
)
