package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	errorRequestIDOffset = headerLength
	errorMessageOffset   = errorRequestIDOffset + 2
	errorDefaultMessage  = "Service error"
)

type Error interface {
	protocol.Message
	ForRequestID() protocol.RequestID
	Unwrap() error
}

type errorMessage struct {
	forId protocol.RequestID
	value error
}

// NewError creates an Error - note that it accepts NoRequest
func NewError(id protocol.RequestID, err error) (Error, error) {
	if err == nil {
		err = protocol.ErrServiceUnavailable
	}
	return &errorMessage{id, err}, nil
}

func (*errorMessage) OpCode() protocol.OpCode            { return protocol.Error }
func (e *errorMessage) ForRequestID() protocol.RequestID { return e.forId }
func (e *errorMessage) Unwrap() error                    { return e.value }
func (e *errorMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	msg, err := protocol.Errors.Marshal(e.value)
	if err != nil {
		return nil, err
	}
	bytes := make([]byte, errorMessageOffset, errorMessageOffset+len(msg))
	injectHeader(bytes, ver, e.OpCode())
	binary.BigEndian.PutUint16(bytes[errorRequestIDOffset:], uint16(e.forId))
	return append(bytes, msg...), nil
}

func (a *errorMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	if len(bytes) < errorMessageOffset {
		return protocol.ErrInvalidLength(errorMessageOffset, len(bytes))
	}

	id := protocol.RequestID(binary.BigEndian.Uint16(bytes[errorRequestIDOffset:]))
	if err := protocol.Errors.Unmarshal(bytes[errorMessageOffset:], &a.value); err != nil {
		return err
	}
	a.forId = id
	return nil
}

func (e *errorMessage) String() string {
	return fmt.Sprintf("<%s> %d %v", e.OpCode(), e.ForRequestID(), e.Unwrap())
}
