package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

type OpaqueBytes = stream.OpaqueBytes

const (
	initRequestIDOffset  = headerLength
	initMethodOffset     = initRequestIDOffset + 2
	initAccessCodeOffset = initMethodOffset + 1
)

// Init is used to establish an initial handshake.
type Init interface {
	protocol.Request
	Version() protocol.Version
	AccessCode() OpaqueBytes
	AuthMethod() stream.AuthMethod
}

type initMessage struct {
	id         protocol.RequestID
	ver        protocol.Version
	auth       stream.AuthMethod
	accessCode OpaqueBytes
}

// NewInit creates an Init
func NewInit(id protocol.RequestID, ver protocol.Version, auth stream.AuthMethod, accessCode OpaqueBytes) (Init, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	if !auth.IsValid() {
		return nil, protocol.ErrInvalidAuthMethod
	}
	return &initMessage{id, ver, auth, accessCode}, nil
}

func (*initMessage) OpCode() protocol.OpCode         { return protocol.Init }
func (i *initMessage) RequestID() protocol.RequestID { return i.id }
func (i *initMessage) Version() protocol.Version     { return i.ver }
func (i *initMessage) AccessCode() OpaqueBytes       { return i.accessCode }
func (i *initMessage) AuthMethod() stream.AuthMethod { return i.auth }
func (i *initMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	bytes := make([]byte, initAccessCodeOffset+len(i.accessCode))
	injectHeader(bytes, ver, i.OpCode())
	binary.BigEndian.PutUint16(bytes[initRequestIDOffset:], uint16(i.id))
	bytes[initMethodOffset] = byte(i.auth)
	copy(bytes[initAccessCodeOffset:], i.accessCode)
	return bytes, nil
}

func (i *initMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}

	if len(bytes) < initAccessCodeOffset {
		return protocol.ErrInvalidLength(initAccessCodeOffset, len(bytes))
	}

	id := protocol.RequestID(binary.BigEndian.Uint16(bytes[initRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	auth := stream.AuthMethod(bytes[initMethodOffset])
	if !auth.IsValid() {
		return protocol.ErrInvalidAuthMethod
	}
	i.id = id
	i.ver = ver
	i.auth = auth
	i.accessCode = OpaqueBytes(bytes[initAccessCodeOffset:])
	return nil
}

func (i *initMessage) String() string {
	return fmt.Sprintf("<%v> %d %v code{%d}", i.OpCode(), i.Version(), i.AuthMethod(), len(i.AccessCode()))
}
